#include "jswrapper.h"

#include <stdlib.h>
#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

JSWrapper *jswrapper_create (const char *filename)
{
  return jswrapper_createrw (filename, "r");
}

JSWrapper *jswrapper_createrw (const char *filename, const char *rw)
{
  JSWrapper *jsw = NULL;

  /* allocate the memory for the JavaSeisWrapper */
  jsw = (JSWrapper *)malloc (sizeof(JSWrapper));
  if (jsw == NULL) {
    return NULL;
  }

  /* initialize the variables in jsw */
  jsw->_jmw = NULL;

  /* instantiate a JMWrapper class */
  jsw->_jmw = jmwrapper_create (
    "org/javaseis/util/access/JavaSeisWrapper");

  if (jsw->_jmw == NULL) {
    jswrapper_delete (jsw);
    return NULL;
  }

  /* set the path */
  if (jmwrapper_chars (jsw->_jmw, "path",
    (char *)filename, strlen(filename), JMW_IN) <= 0) {
    jswrapper_delete (jsw);
    return NULL;
  }

  /* set the read/write mode */
  if (jmwrapper_chars (jsw->_jmw, "rwMode", (char *)rw, strlen(rw),
    JMW_IN) <= 0) {
    jswrapper_delete (jsw);
    return NULL;
  }

  /* initialize the JMWrapper class */
  if (jmwrapper_none (jsw->_jmw, "init") != 1) {
    jswrapper_delete (jsw);
    return NULL;
  }

  return jsw;
}

int jswrapper_isajavaseispath (JSWrapper *jsw)
{
  int retval;

  if (jmwrapper_ints (jsw->_jmw, "isAJavaSeisPath", &retval, 1,
    JMW_OUT) == 0) {
    retval = 0;
  }
  return retval;
}

/* return 1 for good status and 0 otherwise */
int jswrapper_status (JSWrapper *jsw)
{
  int retval;

  if (jmwrapper_ints (jsw->_jmw, "status", &retval, 1,
    JMW_OUT) == 1) {
    if (retval == JSW_NORMAL) {
      retval = 1; /* designates good status */
    }
    else {
      retval = 0; /* designates error status */
    }
  }
  else {
    retval = 0; /* designates error status */
  }
  return retval;
}

char *jswrapper_message (JSWrapper *jsw, char *mess,
  size_t max_size)
{
  char *retval;

  if (jmwrapper_chars (jsw->_jmw, "errorMessage", mess, max_size,
    JMW_OUT) == 1) {
    retval = mess;
  }
  else {
    retval = NULL;
  }
  return retval;
}

long jswrapper_gettracecount (JSWrapper *jsw)
{
  long retval;

  if (jmwrapper_longs (jsw->_jmw, "traceCount", &retval, 1,
    JMW_OUT) == 0) {
    retval = 0;
  }
  return retval;
}

int jswrapper_getnumdimensions (JSWrapper *jsw)
{
  int retval;

  if (jmwrapper_ints (jsw->_jmw, "dimensionCount", &retval, 1,
    JMW_OUT) == 0) {
    retval = 0;
  }
  return retval;
}

int jswrapper_getsamplecount (JSWrapper *jsw)
{
  int retval;

  if (jmwrapper_ints (jsw->_jmw, "sampleCount", &retval, 1,
    JMW_OUT) == 0) {
    retval = 0;
  }
  return retval;
}

int jswrapper_gettrace (JSWrapper *jsw, float *trace,
  size_t max_size)
{
  int retval = jmwrapper_floats (jsw->_jmw, "getTrace", trace,
    max_size, JMW_OUT);
  return retval;
}

int jswrapper_puttrace (JSWrapper *jsw, float *trace,
  size_t max_size)
{
  int retval = jmwrapper_floats (jsw->_jmw, "setTrace", trace,
    max_size, JMW_IN);
  return retval;
}

long jswrapper_gettracenumber (JSWrapper *jsw)
{
  long retval;

  if (jmwrapper_longs (jsw->_jmw, "getTraceNumber", &retval, 1,
    JMW_OUT) == 0) {
    retval = 0;
  }
  return retval;
}

int jswrapper_settracenumber (JSWrapper *jsw, long trace_num)
{
  int retval = jmwrapper_longs (jsw->_jmw, "setTraceNumber",
    &trace_num, 1, JMW_IN);
  return retval;
}

int jswrapper_open (JSWrapper *jsw)
{
  int retval = jmwrapper_none (jsw->_jmw, "open");
  return retval;
}

int jswrapper_setaxis (JSWrapper *jsw, int idim, int length,
  const char *domain, const char *units, long logical_org,
  long logical_del, double physical_org, double physical_del)
{
  long lngs[2];
  double dbls[2];

  lngs[0] = logical_org;
  lngs[1] = logical_del;
  dbls[0] = physical_org;
  dbls[1] = physical_del;

  int retval =
    jmwrapper_ints (jsw->_jmw, "axisDimension", &idim, 1,
      JMW_IN) == 1 &&
    jmwrapper_ints (jsw->_jmw, "setAxisLength", &length, 1,
      JMW_IN) == 1 &&
    jmwrapper_chars (jsw->_jmw, "setAxisDomain", (char *)domain,
      strlen(domain), JMW_IN) == 1 &&
    jmwrapper_chars (jsw->_jmw, "setAxisUnits", (char *)units,
      strlen(units), JMW_IN) == 1 &&
    jmwrapper_longs (jsw->_jmw, "setAxisLogicals", lngs, 2,
      JMW_IN) == 1 && 
    jmwrapper_doubles (jsw->_jmw, "setAxisPhysicals", dbls, 2,
      JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "setAxis") == 1 ? 1 : 0;

  return retval;
}

/* caller's responsibility to make sure that domain and units
 *   both have the capacity to hold 24 chars
 */
int jswrapper_getaxis (JSWrapper *jsw, int idim, int *length,
  char *domain, char *units, long *logical_org, long *logical_del,
  double *physical_org, double *physical_del)
{
  long lngs[2];
  double dbls[2];
  size_t max_size = 24;

  int retval =
    jmwrapper_ints (jsw->_jmw, "axisDimension", &idim, 1,
      JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "getAxis") == 1 &&
    jmwrapper_ints (jsw->_jmw, "getAxisLength", length, 1,
      JMW_OUT) == 1 &&
    jmwrapper_chars (jsw->_jmw, "getAxisDomain", domain, max_size,
      JMW_OUT) == 1 &&
    jmwrapper_chars (jsw->_jmw, "getAxisUnits", units, max_size,
      JMW_OUT) == 1 &&
    jmwrapper_longs (jsw->_jmw, "getAxisLogicals", lngs, 2,
      JMW_OUT) == 1 && 
    jmwrapper_doubles (jsw->_jmw, "getAxisPhysicals", dbls, 2,
      JMW_OUT) == 1 ? 1 : 0;

  if (retval == 1) {
    *logical_org = lngs[0];
    *logical_del = lngs[1];
    *physical_org = dbls[0];
    *physical_del = dbls[1];
  }

  return retval;
}

int jswrapper_addintproperty (JSWrapper *jsw, const char *name,
  const char *description)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_chars (jsw->_jmw, "propertyDescription",
      (char *)description, strlen(description), JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "addIntProperty") == 1 ? 1 : 0;

  return retval;
}

int jswrapper_addfloatproperty (JSWrapper *jsw, const char *name,
  const char *description)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_chars (jsw->_jmw, "propertyDescription",
      (char *)description, strlen(description), JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "addFloatProperty") == 1 ? 1 : 0;

  return retval;
}

int jswrapper_adddoubleproperty (JSWrapper *jsw, const char *name,
  const char *description)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_chars (jsw->_jmw, "propertyDescription",
      (char *)description, strlen(description), JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "addDoubleProperty") == 1 ? 1 : 0;

  return retval;
}

int jswrapper_addlongproperty (JSWrapper *jsw, const char *name,
  const char *description)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_chars (jsw->_jmw, "propertyDescription",
      (char *)description, strlen(description), JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "addLongProperty") == 1 ? 1 : 0;

  return retval;
}

int jswrapper_putintproperty (JSWrapper *jsw, const char *name,
  int value)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_ints (jsw->_jmw, "setIntProperty", &value, 1,
      JMW_IN) == 1 ? 1 : 0;

  return retval;
}

int jswrapper_putfloatproperty (JSWrapper *jsw, const char *name,
  float value)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_floats (jsw->_jmw, "setFloatProperty", &value, 1,
      JMW_IN) == 1 ? 1 : 0;

  return retval;
}

int jswrapper_putdoubleproperty (JSWrapper *jsw, const char *name,
  double value)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_doubles (jsw->_jmw, "setDoubleProperty", &value, 1,
      JMW_IN) == 1 ? 1 : 0;

  return retval;
}

int jswrapper_putlongproperty (JSWrapper *jsw, const char *name,
  long value)
{
  int retval =
    jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
      strlen(name), JMW_IN) == 1 &&
    jmwrapper_longs (jsw->_jmw, "setLongProperty", &value, 1,
      JMW_IN) == 1 ? 1 : 0;

  return retval;
}

int jswrapper_getintproperty (JSWrapper *jsw, const char *name)
{
  int retval;

  if (
    !(jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
        strlen(name), JMW_IN) == 1 &&
      jmwrapper_ints (jsw->_jmw, "getIntProperty", &retval, 1,
        JMW_OUT) == 1)) {
    retval = -999;
  }

  return retval;
}

float jswrapper_getfloatproperty (JSWrapper *jsw, const char *name)
{
  float retval;

  if (
    !(jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
        strlen(name), JMW_IN) == 1 &&
      jmwrapper_floats (jsw->_jmw, "getFloatProperty", &retval, 1,
        JMW_OUT) == 1)) {
    retval = -999;
  }

  return retval;
}

double jswrapper_getdoubleproperty (JSWrapper *jsw,
  const char *name)
{
  double retval;

  if (
    !(jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
        strlen(name), JMW_IN) == 1 &&
      jmwrapper_doubles (jsw->_jmw, "getDoubleProperty", &retval, 1,
        JMW_OUT) == 1)) {
    retval = -999;
  }

  return retval;
}

long jswrapper_getlongproperty (JSWrapper *jsw, const char *name)
{
  long retval;

  if (
    !(jmwrapper_chars (jsw->_jmw, "propertyName", (char *)name,
        strlen(name), JMW_IN) == 1 &&
      jmwrapper_longs (jsw->_jmw, "getLongProperty", &retval, 1,
        JMW_OUT) == 1)) {
    retval = -999;
  }

  return retval;
}

int jswrapper_setdatatype (JSWrapper *jsw, const char *type)
{
  int retval = jmwrapper_chars (jsw->_jmw, "setDataType",
    (char *)type, strlen(type), JMW_IN);

  return retval;
}

char *jswrapper_getdatatype (JSWrapper *jsw, char *type,
  size_t max_size)
{
  char *retval;

  int status = jmwrapper_chars (jsw->_jmw, "getDataType", type,
    max_size, JMW_OUT);

  if (status == 1) {
    retval = type;
  }
  else {
    retval = NULL;
  }
  return retval;
}

int jswrapper_settraceformat (JSWrapper *jsw, const char *format)
{
  int retval = jmwrapper_chars (jsw->_jmw, "setTraceFormat",
    (char *)format, strlen(format), JMW_IN);

  return retval;
}

char *jswrapper_gettraceformat (JSWrapper *jsw, char *format,
  size_t max_size)
{
  char *retval;

  if (jmwrapper_chars (jsw->_jmw, "getTraceFormat", format,
    max_size, JMW_OUT) == 1) {
    retval = format;
  }
  else {
    retval = NULL;
  }
  return retval;
}

int jswrapper_setbin (JSWrapper *jsw, double *world,
  double *grid, long *logical)
{
  int retval =
    jmwrapper_doubles (jsw->_jmw, "setBinWorld", world, 6,
      JMW_IN) == 1 &&
    jmwrapper_doubles (jsw->_jmw, "setBinGrid", grid, 4,
      JMW_IN) == 1 &&
    jmwrapper_longs (jsw->_jmw, "setBinLogical", logical, 8,
      JMW_IN) == 1 &&
    jmwrapper_none (jsw->_jmw, "setBin") == 1 ? 1 : 0;

  return retval;
}

/* caller's responsibility to make sure that world, grid,
 *   logical have dimensions 6, 4, & 8 respectively
 */
int jswrapper_getbin (JSWrapper *jsw, double *world,
  double *grid, long *logical)
{
  int retval =
    jmwrapper_none (jsw->_jmw, "getBin") == 1 &&
    jmwrapper_doubles (jsw->_jmw, "getBinWorld", world, 6,
      JMW_OUT) == 1 &&
    jmwrapper_doubles (jsw->_jmw, "getBinGrid", grid, 4,
      JMW_OUT) == 1 &&
    jmwrapper_longs (jsw->_jmw, "getBinLogical", logical, 8,
      JMW_OUT) == 1 ? 1 : 0;

  return retval;
}

void jswrapper_close (JSWrapper *jsw)
{
  jmwrapper_none (jsw->_jmw, "close");
}

int jswrapper_remove (JSWrapper *jsw)
{
  int retval = jmwrapper_none (jsw->_jmw, "delete");

  return retval;
}

void jswrapper_delete (JSWrapper *jsw)
{
  if (jsw != NULL) {
    jmwrapper_delete (jsw->_jmw);
    free (jsw);
  }
}

#ifdef __cplusplus
}
#endif
