#include "jsfiles.h"
#include "jswrapper.h"

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#ifdef __cplusplus
extern "C" {
#endif

struct _JSFiles
{
  JSWrapper **_jsw;
  int _alloc;
};

typedef struct _JSFiles JSFiles;

static JSFiles *JSFILES = NULL;

#define ALLOC_DEF 20

/*--------------------- private function prototype ------------------------*/
int jsfiles_getnext (void);

/*
 * allow the static structure to expand by not shrink
 * allow the constructor to be reentrant
 * protect previously opened files
 * every open causes a jswrapper class to be created
 * a close will delete a jswrapper class
 * app determines number of files opened simultaneously
 *
 */
int jsfiles_create (int *alloc)
{
  int retval;

  JSFiles *jsfiles = NULL;
  int doit, expand, k2;

  /* make sure the number to allocate is valid */
  if (alloc == NULL || *alloc < 1) {
    if (JSFILES != NULL) {
      assert (JSFILES->_alloc > 0); /* programming sanity check */
      retval = JSFILES->_alloc;
    }
    else {
      retval = ALLOC_DEF;
    }
  }
  else {
    retval = *alloc;
  }

  /* see if requested more */
  if (JSFILES != NULL) {
    if (retval > JSFILES->_alloc) {
      expand = 1;
      doit = 1;
    }
    else {
      expand = 0;
      doit = 0;
    }
  }
  else {
    expand = 0;
    doit = 1;
  }

  if (doit != 0) {
    /* begin the initialization of the new structure */
    jsfiles = (JSFiles *)malloc (sizeof(JSFiles));
    if (jsfiles == NULL) return 0;
    jsfiles->_jsw = (JSWrapper **)malloc ((retval)*sizeof(JSWrapper*));
    for (k2 = 0; k2 < retval; k2++) {
      jsfiles->_jsw[k2] = NULL;
    }
  }

  /* complete initialization */
  if (doit != 0 && expand != 0) {
    jsfiles->_alloc = retval;
    for (k2 = 0; k2 < JSFILES->_alloc; k2++) {
      jsfiles->_jsw[k2] = JSFILES->_jsw[k2];
    }
    /* do not close the opened files, but free old structure */
    free (JSFILES->_jsw);
    JSFILES->_jsw = NULL;
    free (JSFILES);
    JSFILES = jsfiles;
  }
  else if (doit != 0 && JSFILES == NULL) {
    jsfiles->_alloc = retval;
    JSFILES = jsfiles;
  }

  /* do a programming sanity check */
  assert (JSFILES != NULL);
  assert (JSFILES->_alloc == retval && retval > 0);

  return retval;
}

/* create the static structure but do not open yet */
int jsfiles_getlun (const char *filename, const char *rw)
{
  int retval, current;
  JSWrapper *obj;

  if (JSFILES == NULL) assert (jsfiles_create(NULL) > 0);

  current = jsfiles_getnext ();
  if (current < JSFILES->_alloc) {
    obj = jswrapper_createrw (filename, rw);
    if (obj != NULL) {
      JSFILES->_jsw[current] = obj;
      retval = current + 1; /* return a 1-relative number */
    }
    else {
      retval = 0;
    }
  }
  else {
    /* no more JSFILES space left */
    retval = 0;
  }
  return retval;
}

/* open the given lun */
int jsfiles_open (int which)
{
  int retval;
  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_open (obj);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_close (int which)
{
  int retval;

  int current = which - 1;
  JSWrapper *obj;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      jswrapper_close (obj);
      jswrapper_delete (obj);
      JSFILES->_jsw[current] = NULL;
      retval = 1;
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_isa (int which)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_isajavaseispath (obj);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_status (int which)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_status (obj);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_message (int which, char *mess, int max_size)
{
  int retval;
  JSWrapper *obj;
  char *mes;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      mes = jswrapper_message (obj, mess, (size_t)max_size);
      if (mes != NULL) {
        retval = strlen (mess);
      }
      else {
        retval = 0;
      }
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

long jsfiles_gettracecount (int which)
{
  long retval;

  JSWrapper *obj;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_gettracecount (obj);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_getnumdimensions (int which)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getnumdimensions (obj);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_getsamplecount (int which)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getsamplecount (obj);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_gettrace (int which, float *trace, int max_size)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_gettrace (obj, trace, (size_t)max_size);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_puttrace (int which, float *trace, int size)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_puttrace (obj, trace, (size_t)size);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_settracenumber (int which, long trace_num)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_settracenumber (obj, trace_num);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_setaxis (int which, int idim, int length, const char *domain,
  const char *units, long logical_org, long logical_del, double physical_org,
  double physical_del)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_setaxis (obj, idim-1, length, domain, units,
        logical_org, logical_del, physical_org, physical_del);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_getaxis (int which, int idim, int *length, char *domain,
  char *units, long *logical_org, long *logical_del, double *physical_org,
  double *physical_del)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getaxis (obj, idim-1, length, domain, units,
        logical_org, logical_del, physical_org, physical_del);
    }
    else {
      printf ("jsfiles_getaxis: obj == NULL\n");
      retval = 0;
    }
  }
  else {
    printf ("jsfiles_getaxis: current = %d\n", current);
    retval = 0;
  }
  return retval;
}

int jsfiles_addintproperty (int which, const char *name,
  const char *description)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_addintproperty (obj, name, description);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_addfloatproperty (int which, const char *name,
  const char *description)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_addfloatproperty (obj, name, description);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_adddoubleproperty (int which, const char *name,
  const char *description)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_adddoubleproperty (obj, name, description);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_addlongproperty (int which, const char *name,
  const char *description)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_addlongproperty (obj, name, description);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_putintproperty (int which, const char *name, int value)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_putintproperty (obj, name, value);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_putfloatproperty (int which, const char *name, float value)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_putfloatproperty (obj, name, value);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_putdoubleproperty (int which, const char *name, double value)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_putdoubleproperty (obj, name, value);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_putlongproperty (int which, const char *name, long value)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_putlongproperty (obj, name, value);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_getintproperty (int which, const char *name)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getintproperty (obj, name);
    }
    else {
      retval = -999;
    }
  }
  else {
    retval = -999;
  }
  return retval;
}

float jsfiles_getfloatproperty (int which, const char *name)
{
  float retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getfloatproperty (obj, name);
    }
    else {
      retval = -999;
    }
  }
  else {
    retval = -999;
  }
  return retval;
}

double jsfiles_getdoubleproperty (int which, const char *name)
{
  double retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getdoubleproperty (obj, name);
    }
    else {
      retval = -999;
    }
  }
  else {
    retval = -999;
  }
  return retval;
}

long jsfiles_getlongproperty (int which, const char *name)
{
  long retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getlongproperty (obj, name);
    }
    else {
      retval = -999;
    }
  }
  else {
    retval = -999;
  }
  return retval;
}

int jsfiles_setdatatype (int which, const char *type)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_setdatatype (obj, type);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_getdatatype (int which, char *type, size_t max_size)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      jswrapper_getdatatype (obj, type, max_size);
      retval = strlen (type);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_settraceformat (int which, const char *format)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_settraceformat (obj, format);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_gettraceformat (int which, char *format, size_t max_size)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      jswrapper_gettraceformat (obj, format, max_size);
      retval = strlen (format);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_setbin (int which, double *world, double *grid, long *logical)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_setbin (obj, world, grid, logical);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_getbin (int which, double *world, double *grid, long *logical)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;

  assert (JSFILES != NULL);

  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_getbin (obj, world, grid, logical);
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

int jsfiles_remove (int which)
{
  int retval;

  JSWrapper *obj;
  int current = which - 1;
  assert (JSFILES != NULL);
  if (current > -1 && current < JSFILES->_alloc) {
    obj = JSFILES->_jsw[current];
    if (obj != NULL) {
      retval = jswrapper_remove (obj);
      if (retval == 1) {
	jswrapper_delete (obj);
      }
      if (retval == 1) {
	JSFILES->_jsw[current] = NULL;
      }
    }
    else {
      retval = 0;
    }
  }
  else {
    retval = 0;
  }
  return retval;
}

void jsfiles_delete (void)
{
  int k2;

  if (JSFILES != NULL) {
    for (k2 = 0; k2 < JSFILES->_alloc; k2++) {
      jsfiles_close (k2+1);
    }
    free (JSFILES->_jsw);
    JSFILES->_jsw = NULL;
    free (JSFILES);
    JSFILES = NULL;
  }
}

int jsfiles_getnext (void)
{
  int retval, k2;

  for (k2 = 0, retval = JSFILES->_alloc; k2 < JSFILES->_alloc; k2++) {
    if (JSFILES->_jsw[k2] == NULL) {
      retval = k2;
      k2 = JSFILES->_alloc;
    }
  }
  return retval;
}

#ifdef __cplusplus
}
#endif
