/*
 * Decompiled with CFR 0.152.
 */
package org.javaseis.seiszip;

import java.nio.IntBuffer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.DataFormatException;
import org.javaseis.seiszip.BlockCompressor;
import org.javaseis.seiszip.CompressedData;
import org.javaseis.seiszip.DataCorruptedException;
import org.javaseis.seiszip.HdrCompressor;
import org.javaseis.seiszip.Transformer;
import org.javaseis.util.ArrayUtil;

public class SeisPEG {
    private static final Logger LOG = Logger.getLogger("org.javaseis.seiszip");
    private static int c_nThreads = 4;
    private static final int FORWARD = 1;
    private static final int REVERSE = -1;
    private static final int IND_COOKIE = 0;
    private static final int IND_DISTORTION = 1;
    private static final int IND_N1 = 2;
    private static final int IND_N2 = 3;
    private static final int IND_VBLOCK_SIZE = 4;
    private static final int IND_HBLOCK_SIZE = 5;
    private static final int IND_VTRANS_LEN = 6;
    private static final int IND_HTRANS_LEN = 7;
    private static final int IND_NBYTES_TRACES = 8;
    private static final int IND_NBYTES_HDRS = 9;
    private static final int IND_FT_GAIN = 10;
    static final int LEN_HDR_INFO = 11;
    private static final int CACHE_SIZE = 32;
    private static final short COOKIE_V2 = 30607;
    private static final short COOKIE_V3 = 30744;
    private static final short BAD_AMPLITUDE_COOKIE_V2 = 29899;
    private static final short BAD_AMPLITUDE_COOKIE_V3 = 29941;
    private static final int SIZEOF_INT = 4;
    private static final int SIZEOF_FLOAT = 4;
    private static final int SIZEOF_SHORT = 2;
    private static final int SIZEOF_CHAR = 1;
    private int _n1;
    private int _n2;
    private float _distortion;
    private float _ftGainExponent;
    private boolean _ftGainExponentWasStored = false;
    private float[] _ftGain = null;
    private int _verticalBlockSize;
    private int _horizontalBlockSize;
    private int _verticalTransLength;
    private int _horizontalTransLength;
    private int _paddedN1;
    private int _paddedN2;
    private float[] _workBuffer1 = null;
    private float[] _workBlock = null;
    private byte[] _workBuffer2 = null;
    private int _workBuffer2Size = 0;
    private byte[] _compressedBuffer = null;
    private Transformer _transformer = new Transformer();
    private BlockCompressor _blockCompressor;
    private HdrCompressor _hdrCompressor = new HdrCompressor();
    private float[][] _vecW = null;
    private float[] _scratch1 = null;
    private float[] _scratch2 = null;
    private TimeTransformer[] _timeTransformers = null;
    private X1Transformer[] _x1Transformers = null;
    private int _nEnsemblesChecked = 0;
    private int[] _hdrInfo = new int[11];
    private int _traceLength = 0;
    private int _hdrLength = 0;
    private long _nTracesWrittenTotal = 0L;
    private long _nBytesTotal = 0L;

    private static int computePaddedLength(int nsamples, int blockSize) {
        if (blockSize == 0) {
            throw new IllegalArgumentException("Block Size is Invalid");
        }
        int n = nsamples / blockSize * blockSize;
        if (n < nsamples) {
            n += blockSize;
        }
        return n;
    }

    private static int computeBlockSize(int nsamples, Policy policy) {
        int blockSize;
        if (nsamples <= 8) {
            return 8;
        }
        if (nsamples <= 16) {
            return 16;
        }
        if (nsamples <= 24) {
            return 24;
        }
        if (nsamples <= 32) {
            return 32;
        }
        if (policy.equals((Object)Policy.FASTEST)) {
            if (nsamples <= 48) {
                return 24;
            }
            if (nsamples <= 64) {
                return 32;
            }
            return 64;
        }
        int nBlocks = nsamples / 16;
        if (nBlocks * 16 < nsamples) {
            ++nBlocks;
        }
        if ((blockSize = nBlocks * 16) > 512) {
            blockSize = 512;
        }
        return blockSize;
    }

    private static int computeTransLength(int blockSize, Policy policy) {
        if (policy.equals((Object)Policy.FASTEST)) {
            return 8;
        }
        if (blockSize / 16 * 16 == blockSize) {
            return 16;
        }
        return 8;
    }

    private static void checkBlockSize(int blockSize) {
        if (blockSize / 8 * 8 != blockSize) {
            throw new IllegalArgumentException("Block size of " + blockSize + " is invalid");
        }
    }

    private static void checkTransLength(int transLength, int blockSize) {
        int nsubBlocks;
        if (transLength == 8) {
            int nsubBlocks2 = blockSize / 8;
            if (nsubBlocks2 * 8 != blockSize) {
                throw new IllegalArgumentException("Invalid transform length of " + transLength + " for block size of 8");
            }
        } else if (transLength == 16 && (nsubBlocks = blockSize / 16) * 16 != blockSize) {
            throw new IllegalArgumentException("Invalid transform length of " + transLength + " for block size of 16");
        }
    }

    private static void fillBuffer(int direction, float[][] traces, int n1, int n2, int paddedN1, int paddedN2, float[] workBuffer, float[] ftGain) {
        int i;
        int j;
        int index = 0;
        for (j = 0; j < n2; ++j) {
            if (direction == 1) {
                ArrayUtil.arraycopy(traces[j], 0, workBuffer, index, n1);
                if (ftGain != null) {
                    SeisPEG.applyFtGain(ftGain, workBuffer, index, n1);
                }
                for (i = n1; i < paddedN1; ++i) {
                    workBuffer[index + i] = 0.0f;
                }
            } else {
                ArrayUtil.arraycopy(workBuffer, index, traces[j], 0, n1);
                if (ftGain != null) {
                    SeisPEG.removeFtGain(ftGain, traces[j]);
                }
            }
            index += paddedN1;
        }
        if (direction == 1) {
            for (j = n2; j < paddedN2; ++j) {
                for (i = 0; i < paddedN1; ++i) {
                    workBuffer[index + i] = 0.0f;
                }
                index += paddedN1;
            }
        }
    }

    public SeisPEG() {
        this._blockCompressor = new BlockCompressor();
    }

    SeisPEG(int[] huffCount, int n1, int n2, float distortion, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength) {
        this._blockCompressor = new BlockCompressor(huffCount);
        float ftGainExponent = 0.0f;
        this.init(n1, n2, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, "full");
    }

    public SeisPEG(int n1, int n2, float distortion, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength) {
        this._blockCompressor = new BlockCompressor();
        float ftGainExponent = 0.0f;
        this.init(n1, n2, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, "full");
    }

    public SeisPEG(int n1, int n2, float distortion, Policy policy) {
        int verticalBlockSize = SeisPEG.computeBlockSize(n1, policy);
        int verticalTransLength = SeisPEG.computeTransLength(verticalBlockSize, policy);
        int horizontalBlockSize = SeisPEG.computeBlockSize(n2, policy);
        int horizontalTransLength = SeisPEG.computeTransLength(horizontalBlockSize, policy);
        this._blockCompressor = new BlockCompressor();
        float ftGainExponent = 0.0f;
        this.init(n1, n2, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, "short");
    }

    public SeisPEG(CompressedData compressedData) {
        this(compressedData.getData());
    }

    public SeisPEG(byte[] compressedByteData) {
        SeisPEG.decodeHdr(compressedByteData, this._hdrInfo);
        float distortion = Float.intBitsToFloat(this._hdrInfo[1]);
        int n1 = this._hdrInfo[2];
        int n2 = this._hdrInfo[3];
        int verticalBlockSize = this._hdrInfo[4];
        int horizontalBlockSize = this._hdrInfo[5];
        int verticalTransLength = this._hdrInfo[6];
        int horizontalTransLength = this._hdrInfo[7];
        float ftGainExponent = Float.intBitsToFloat(this._hdrInfo[10]);
        this._blockCompressor = new BlockCompressor();
        this.init(n1, n2, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, "existing");
    }

    private void init(int n1, int n2, float distortion, float ftGainExponent, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength, String whichConstructor) {
        LOG.fine("SeisPEG.init " + whichConstructor + ": n1=" + n1 + " n2=" + n2);
        if (n1 < 1 || n2 < 1) {
            throw new IllegalArgumentException("The data size must be non-zero");
        }
        if (distortion <= 0.0f) {
            throw new IllegalArgumentException("The distortion is less than or equal to zero");
        }
        SeisPEG.checkBlockSize(verticalBlockSize);
        SeisPEG.checkBlockSize(horizontalBlockSize);
        SeisPEG.checkTransLength(verticalTransLength, verticalBlockSize);
        SeisPEG.checkTransLength(horizontalTransLength, horizontalBlockSize);
        this._n1 = n1;
        this._n2 = n2;
        this._distortion = distortion;
        this._ftGainExponent = ftGainExponent;
        this._verticalBlockSize = verticalBlockSize;
        this._horizontalBlockSize = horizontalBlockSize;
        this._verticalTransLength = verticalTransLength;
        this._horizontalTransLength = horizontalTransLength;
        this._paddedN1 = SeisPEG.computePaddedLength(n1, verticalBlockSize);
        this._paddedN2 = SeisPEG.computePaddedLength(n2, horizontalBlockSize);
        if (c_nThreads > 1) {
            this._timeTransformers = new TimeTransformer[c_nThreads];
            this._x1Transformers = new X1Transformer[c_nThreads];
            for (int i = 0; i < c_nThreads; ++i) {
                this._timeTransformers[i] = new TimeTransformer(i, c_nThreads);
                this._timeTransformers[i].start();
                this._x1Transformers[i] = new X1Transformer(i, c_nThreads);
                this._x1Transformers[i].start();
            }
        }
    }

    public void setGainExponent(float ftGainExponent) {
        if (this._ftGainExponentWasStored && this._ftGainExponent != ftGainExponent) {
            throw new IllegalStateException("Attempt to change ftGainExponent after it has been stored");
        }
        this._ftGainExponent = ftGainExponent;
    }

    public void setDelta(float delta) {
        this._blockCompressor.setDelta(delta);
    }

    public float getDistortion() {
        return this._distortion;
    }

    public int getSamplesPerTrace() {
        return this._n1;
    }

    public int getMaxTracesPerFrame() {
        return this._n1;
    }

    public int getVerticalBlockSize() {
        return this._verticalBlockSize;
    }

    public int getHorizontalBlockSize() {
        return this._horizontalBlockSize;
    }

    public int getVerticalTransLength() {
        return this._verticalTransLength;
    }

    public int getHorizontalTransLength() {
        return this._horizontalTransLength;
    }

    public CompressedData compressedBufferAlloc() {
        return new CompressedData(this.compressedByteBufferAlloc(), 0);
    }

    private byte[] compressedByteBufferAlloc() {
        int nbytes = (double)this._distortion > 0.1 ? this._paddedN1 * this._paddedN2 * 4 / 2 : ((double)this._distortion > 0.01 ? this._paddedN1 * this._paddedN2 * 4 : this._paddedN1 * this._paddedN2 * 4 * 2);
        return new byte[nbytes];
    }

    public float[][] uncompressedBufferAlloc() {
        return new float[this._n2][this._n1];
    }

    public void transform(float[][] traces, int nTraces) {
        if (traces.length > this._n2) {
            throw new IllegalArgumentException("The size of the data cannot increase");
        }
        if (traces[0].length != this._n1) {
            throw new IllegalArgumentException("The size of the data cannot vary (1)");
        }
        if (this._workBuffer1 == null) {
            this._workBuffer1 = new float[this._paddedN1 * this._paddedN2];
        }
        SeisPEG.fillBuffer(1, traces, this._n1, nTraces, this._paddedN1, this._paddedN2, this._workBuffer1, null);
        this.transform2D(this._workBuffer1);
        SeisPEG.fillBuffer(-1, traces, this._n1, nTraces, this._paddedN1, this._paddedN2, this._workBuffer1, null);
    }

    private void transform2D(float[] paddedTraces) {
        this.x1Transform(1, paddedTraces);
        this.timeTransform(1, paddedTraces);
    }

    private boolean containsBadAmplitude(float[][] traces, int nTraces) {
        ++this._nEnsemblesChecked;
        for (int j = 0; j < nTraces; ++j) {
            float[] trace = traces[j];
            for (int i = 0; i < trace.length; ++i) {
                if (!((double)trace[i] < -1.0E15) && !((double)trace[i] > 1.0E15)) continue;
                LOG.warning("Found uncompressible bad amplitude " + trace[i] + " at ensemble " + this._nEnsemblesChecked);
                return true;
            }
        }
        return false;
    }

    public int compress(float[][] traces, int nTraces, byte[] outputData) {
        if (traces == null || outputData == null) {
            throw new IllegalArgumentException("Null arguments are not allowed");
        }
        if (traces.length > this._n2) {
            throw new IllegalArgumentException("The size of the data cannot increase");
        }
        if (traces[0].length != this._n1) {
            throw new IllegalArgumentException("The size of the data cannot vary (2)");
        }
        if (outputData.length < traces.length * traces[0].length * 4) {
            throw new IllegalArgumentException("The output buffer must be large enough to hold an uncompressed copy of the input");
        }
        if (this.containsBadAmplitude(traces, nTraces)) {
            return this.badAmplitudeCompress(traces, nTraces, outputData);
        }
        if (this._ftGainExponent != 0.0f && this._ftGain == null) {
            this._ftGain = this.computeFtGain(this._n1, this._ftGainExponent);
        }
        if (this._workBuffer1 == null) {
            this._workBuffer1 = new float[this._paddedN1 * this._paddedN2];
        }
        SeisPEG.fillBuffer(1, traces, this._n1, nTraces, this._paddedN1, this._paddedN2, this._workBuffer1, this._ftGain);
        int nbytes = this.compress2D(this._workBuffer1, this._distortion, this._ftGainExponent, outputData, outputData.length);
        this._ftGainExponentWasStored = true;
        if (nbytes > outputData.length) {
            throw new RuntimeException("nbytes > outputData.length");
        }
        if (nbytes == 0) {
            return this.badAmplitudeCompress(traces, nTraces, outputData);
        }
        return nbytes;
    }

    private CompressedData badAmplitudeCompress(float[][] traces, int nTraces, CompressedData compressedData) {
        byte[] outputData;
        int size = traces[0].length * nTraces * 4;
        if (compressedData == null) {
            compressedData = new CompressedData(new byte[size], 0);
        }
        if ((outputData = compressedData.getData()).length < size) {
            compressedData = new CompressedData(new byte[size], 0);
        }
        outputData = compressedData.getData();
        int index = 0;
        for (int j = 0; j < nTraces; ++j) {
            float[] trace = traces[j];
            for (int i = 0; i < trace.length; ++i) {
                int ival = Float.floatToIntBits(trace[i]);
                BlockCompressor.stuffIntInBytes(ival, outputData, index);
                index += 4;
            }
        }
        short cookie = (double)this._ftGainExponent == 0.0 ? (short)29899 : 29941;
        SeisPEG.encodeHdr(outputData, cookie, this._distortion, this._ftGainExponent, this._n1, this._n2, this._verticalBlockSize, this._horizontalBlockSize, this._verticalTransLength, this._horizontalTransLength, size, 0);
        this._ftGainExponentWasStored = true;
        compressedData.setDataLength(size);
        return compressedData;
    }

    private int badAmplitudeCompress(float[][] traces, int nTraces, byte[] outputData) {
        if (outputData.length < traces.length * traces[0].length * 4) {
            throw new IllegalArgumentException("The output buffer must be large enough to hold an uncompressed copy of the input");
        }
        int index = 0;
        for (int j = 0; j < nTraces; ++j) {
            float[] trace = traces[j];
            for (int i = 0; i < trace.length; ++i) {
                int ival = Float.floatToIntBits(trace[i]);
                BlockCompressor.stuffIntInBytes(ival, outputData, index);
                index += 4;
            }
        }
        int size = traces[0].length * nTraces * 4;
        short cookie = (double)this._ftGainExponent == 0.0 ? (short)29899 : 29941;
        SeisPEG.encodeHdr(outputData, cookie, this._distortion, this._ftGainExponent, this._n1, this._n2, this._verticalBlockSize, this._horizontalBlockSize, this._verticalTransLength, this._horizontalTransLength, size, 0);
        this._ftGainExponentWasStored = true;
        return size;
    }

    private void badAmplitudeUncompress(byte[] inData, int inDataLength, float[][] traces) {
        int nTraces = inDataLength / 4 / traces[0].length;
        if (nTraces < 1) {
            throw new IllegalArgumentException("Input data is impossibly short");
        }
        if (nTraces > traces.length) {
            throw new IllegalArgumentException("Input data is impossibly long " + nTraces + ">" + traces.length);
        }
        int index = 0;
        for (int j = 0; j < nTraces; ++j) {
            int i;
            float[] trace = traces[j];
            for (i = 0; i < trace.length; ++i) {
                int ival = BlockCompressor.stuffBytesInInt(inData, index);
                trace[i] = Float.intBitsToFloat(ival);
                index += 4;
            }
            if (j != 0) continue;
            for (i = 0; i < Math.min(11, trace.length); ++i) {
                trace[i] = 0.0f;
            }
        }
    }

    public CompressedData compress(float[][] traces, CompressedData compressedData) {
        return this.compress(traces, traces.length, compressedData);
    }

    public CompressedData compress(float[][] traces, int nTraces, CompressedData compressedData) {
        if (traces.length > this._n2) {
            throw new IllegalArgumentException("The size of the data cannot increase");
        }
        if (traces[0].length != this._n1) {
            throw new IllegalArgumentException("The size of the data cannot vary (3)");
        }
        if (this.containsBadAmplitude(traces, nTraces)) {
            return this.badAmplitudeCompress(traces, nTraces, compressedData);
        }
        if (this._ftGainExponent != 0.0f && this._ftGain == null) {
            this._ftGain = this.computeFtGain(this._n1, this._ftGainExponent);
        }
        if (this._workBuffer1 == null) {
            this._workBuffer1 = new float[this._paddedN1 * this._paddedN2];
        }
        SeisPEG.fillBuffer(1, traces, this._n1, nTraces, this._paddedN1, this._paddedN2, this._workBuffer1, this._ftGain);
        if (compressedData == null) {
            compressedData = this.compressedBufferAlloc();
        }
        int nbytes = 0;
        while (nbytes == 0) {
            byte[] outputData = compressedData.getData();
            nbytes = this.compress2D(this._workBuffer1, this._distortion, this._ftGainExponent, outputData, outputData.length);
            this._ftGainExponentWasStored = true;
            if (nbytes > outputData.length) {
                throw new RuntimeException("nbytes > outputData.length");
            }
            if (nbytes == 0) {
                LOG.warning("Initial buffer was too small - compression must be repeated with a larger output buffer");
                if (LOG.isLoggable(Level.FINE)) {
                    Thread.dumpStack();
                }
                SeisPEG.fillBuffer(1, traces, this._n1, nTraces, this._paddedN1, this._paddedN2, this._workBuffer1, this._ftGain);
                int newLength = outputData.length * 2;
                compressedData = new CompressedData(new byte[newLength], 0);
                continue;
            }
            compressedData.setDataLength(nbytes);
            return compressedData;
        }
        return null;
    }

    public float[][] difference(float[][] traces) throws DataCorruptedException {
        CompressedData compressedData = this.compressedBufferAlloc();
        this.compress(traces, compressedData);
        float[][] uncompressedData = this.uncompressedBufferAlloc();
        this.uncompress(compressedData, uncompressedData);
        for (int j = 0; j < traces.length; ++j) {
            for (int i = 0; i < traces[0].length; ++i) {
                float[] fArray = uncompressedData[j];
                int n = i;
                fArray[n] = fArray[n] - traces[j][i];
            }
        }
        return uncompressedData;
    }

    public void uncompress(byte[] compressedByteData, int compressedDataLength, float[][] traces) throws DataCorruptedException {
        int iflag;
        if (traces.length != this._n2 || traces[0].length != this._n1) {
            throw new IllegalArgumentException("The size of the data cannot vary (4) " + traces.length + " " + this._n2 + " " + traces[0].length + " " + this._n1);
        }
        if (SeisPEG.badAmplitudeData(compressedByteData)) {
            SeisPEG.decodeHdr(compressedByteData, this._hdrInfo);
            int nBytesTraces = this._hdrInfo[8];
            this.badAmplitudeUncompress(compressedByteData, nBytesTraces, traces);
            return;
        }
        if (this._ftGainExponent != 0.0f && this._ftGain == null) {
            this._ftGain = this.computeFtGain(this._n1, this._ftGainExponent);
        }
        if (this._workBuffer1 == null) {
            this._workBuffer1 = new float[this._paddedN1 * this._paddedN2];
        }
        if ((iflag = this.uncompress2D(compressedByteData, compressedDataLength, this._workBuffer1)) != 0) {
            throw new DataCorruptedException("Compressed data is corrupted");
        }
        SeisPEG.fillBuffer(-1, traces, this._n1, this._n2, this._paddedN1, this._paddedN2, this._workBuffer1, this._ftGain);
    }

    public void uncompress(CompressedData compressedData, float[][] traces) throws DataCorruptedException {
        int iflag;
        if (traces.length != this._n2 || traces[0].length != this._n1) {
            throw new IllegalArgumentException("The size of the data cannot vary (5) " + traces.length + " " + this._n2 + " " + traces[0].length + " " + this._n1);
        }
        byte[] inData = compressedData.getData();
        if (SeisPEG.badAmplitudeData(inData)) {
            this.badAmplitudeUncompress(inData, compressedData.getDataLength(), traces);
            return;
        }
        if (this._ftGainExponent != 0.0f && this._ftGain == null) {
            this._ftGain = this.computeFtGain(this._n1, this._ftGainExponent);
        }
        if (this._workBuffer1 == null) {
            this._workBuffer1 = new float[this._paddedN1 * this._paddedN2];
        }
        if ((iflag = this.uncompress2D(inData, compressedData.getDataLength(), this._workBuffer1)) != 0) {
            throw new DataCorruptedException("Compressed data is corrupted");
        }
        SeisPEG.fillBuffer(-1, traces, this._n1, this._n2, this._paddedN1, this._paddedN2, this._workBuffer1, this._ftGain);
    }

    private void timeTransform(int direction, float[] paddedTraces) {
        if (this._timeTransformers != null) {
            this.timeTransformThreaded(direction, paddedTraces);
            return;
        }
        int nblocksVertical = this._paddedN1 / this._verticalBlockSize;
        assert (nblocksVertical * this._verticalBlockSize == this._paddedN1);
        if (this._scratch1 == null) {
            this._scratch1 = new float[this._paddedN1 + this._verticalBlockSize];
        }
        for (int j = 0; j < this._paddedN2; ++j) {
            int index = j * this._paddedN1;
            if (direction == 1) {
                this._transformer.lotFwd(paddedTraces, index, this._verticalBlockSize, this._verticalTransLength, nblocksVertical, this._scratch1);
                continue;
            }
            this._transformer.lotRev(paddedTraces, index, this._verticalBlockSize, this._verticalTransLength, nblocksVertical, this._scratch1);
        }
    }

    private void timeTransformThreaded(int direction, float[] paddedTraces) {
        int i;
        for (i = 0; i < this._timeTransformers.length; ++i) {
            this._timeTransformers[i].setData(direction, paddedTraces);
        }
        for (i = 0; i < this._timeTransformers.length; ++i) {
            this._timeTransformers[i].waitForFinish();
        }
    }

    private void x1Transform(int direction, float[] paddedTraces) {
        if (this._x1Transformers != null) {
            this.x1TransformThreaded(direction, paddedTraces);
            return;
        }
        int nblocksHorizontal = this._paddedN2 / this._horizontalBlockSize;
        assert (nblocksHorizontal * this._horizontalBlockSize == this._paddedN2);
        if (this._vecW == null) {
            this._vecW = new float[32][this._paddedN2];
        }
        if (this._scratch2 == null) {
            this._scratch2 = new float[this._paddedN2 + this._horizontalBlockSize];
        }
        int nsamps = this._paddedN1;
        for (int i = 0; i < nsamps; i += 32) {
            int l;
            int l2;
            int index;
            int m;
            int n = nsamps - i;
            if (n > 32) {
                n = 32;
            }
            for (m = 0; m < this._paddedN2; ++m) {
                index = m * this._paddedN1 + i;
                for (l2 = 0; l2 < n; ++l2) {
                    this._vecW[l2][m] = paddedTraces[l2 + index];
                }
            }
            if (direction == 1) {
                for (l = 0; l < n; ++l) {
                    this._transformer.lotFwd(this._vecW[l], 0, this._horizontalBlockSize, this._horizontalTransLength, nblocksHorizontal, this._scratch2);
                }
            } else {
                for (l = 0; l < n; ++l) {
                    this._transformer.lotRev(this._vecW[l], 0, this._horizontalBlockSize, this._horizontalTransLength, nblocksHorizontal, this._scratch2);
                }
            }
            for (m = 0; m < this._paddedN2; ++m) {
                index = m * this._paddedN1 + i;
                for (l2 = 0; l2 < n; ++l2) {
                    paddedTraces[l2 + index] = this._vecW[l2][m];
                }
            }
        }
    }

    private void x1TransformThreaded(int direction, float[] paddedTraces) {
        int i;
        for (i = 0; i < this._x1Transformers.length; ++i) {
            this._x1Transformers[i].setData(direction, paddedTraces);
        }
        for (i = 0; i < this._x1Transformers.length; ++i) {
            this._x1Transformers[i].waitForFinish();
        }
    }

    private int codeAllBlocks(int direction, float[] paddedTraces, int paddedN1, int paddedN2, float distortion, int verticalBlockSize, int horizontalBlockSize, byte[] encodedData, int index, int bufferSize) {
        int nblocksVertical = paddedN1 / verticalBlockSize;
        int nblocksHorizontal = paddedN2 / horizontalBlockSize;
        if (nblocksVertical < 1 || nblocksHorizontal < 1) {
            throw new RuntimeException("Padded data size is less than 1 block");
        }
        int samplesPerBlock = verticalBlockSize * horizontalBlockSize;
        int samplesPerColumn = samplesPerBlock * nblocksVertical;
        if (this._workBlock == null) {
            this._workBlock = new float[verticalBlockSize * horizontalBlockSize];
        }
        if (this._workBuffer2 == null) {
            this._workBuffer2Size = verticalBlockSize * horizontalBlockSize * 4;
            this._workBuffer2 = new byte[this._workBuffer2Size];
        }
        int nbytesTotal = 0;
        int encodedDataIndex = index;
        int workBlockIndex = 0;
        int dataIndex = 0;
        for (int l = 0; l < nblocksHorizontal; ++l) {
            for (int k = 0; k < nblocksVertical; ++k) {
                dataIndex = l * samplesPerColumn + k * verticalBlockSize;
                workBlockIndex = 0;
                int nbytes = 0;
                if (direction == -1) {
                    nbytes = BlockCompressor.stuffBytesInInt(encodedData, encodedDataIndex);
                    if (encodedDataIndex - index + nbytes > bufferSize) {
                        if (LOG.isLoggable(Level.FINE)) {
                            LOG.fine("encodedDataIndex-index)+nbytes > bufferSize");
                        }
                        return 0;
                    }
                    int ierr = -1;
                    while (ierr != 0) {
                        ierr = this._blockCompressor.dataDecode(encodedData, 4 + encodedDataIndex, this._workBuffer2, this._workBuffer2Size, samplesPerBlock, this._workBlock);
                        if (ierr == 0) continue;
                        this._workBuffer2Size *= 2;
                        this._workBuffer2 = new byte[this._workBuffer2Size];
                    }
                    encodedDataIndex += nbytes;
                }
                for (int j = 0; j < horizontalBlockSize; ++j) {
                    int i;
                    if (direction == 1) {
                        for (i = 0; i < verticalBlockSize; ++i) {
                            this._workBlock[i + workBlockIndex] = paddedTraces[i + dataIndex];
                        }
                    } else {
                        for (i = 0; i < verticalBlockSize; ++i) {
                            paddedTraces[i + dataIndex] = this._workBlock[i + workBlockIndex];
                        }
                    }
                    dataIndex += paddedN1;
                    workBlockIndex += verticalBlockSize;
                }
                if (direction == 1) {
                    int nbytesAvailable = bufferSize - (encodedDataIndex + 4 - index);
                    if (nbytesAvailable < 1) {
                        if (LOG.isLoggable(Level.FINE)) {
                            LOG.fine("nbytesAvailable < 1");
                        }
                        return 0;
                    }
                    nbytes = this._blockCompressor.dataEncode(this._workBlock, samplesPerBlock, distortion, encodedData, encodedDataIndex + 4, nbytesAvailable);
                    if (nbytes == 0) {
                        if (LOG.isLoggable(Level.FINE)) {
                            LOG.fine("nbytesAvailable == 0");
                        }
                        return 0;
                    }
                    BlockCompressor.stuffIntInBytes(nbytes += 4, encodedData, encodedDataIndex);
                    encodedDataIndex += nbytes;
                }
                nbytesTotal += nbytes;
            }
        }
        return nbytesTotal;
    }

    private static int updateHdr(byte[] encodedData, int[] hdrInfo) {
        short cookie = (short)hdrInfo[0];
        float distortion = Float.intBitsToFloat(hdrInfo[1]);
        int n1 = hdrInfo[2];
        int n2 = hdrInfo[3];
        int verticalBlockSize = hdrInfo[4];
        int horizontalBlockSize = hdrInfo[5];
        int verticalTransLength = hdrInfo[6];
        int horizontalTransLength = hdrInfo[7];
        int nBytesTraces = hdrInfo[8];
        int nBytesHdrs = hdrInfo[9];
        float ftGainExponent = Float.intBitsToFloat(hdrInfo[10]);
        return SeisPEG.encodeHdr(encodedData, cookie, distortion, ftGainExponent, n1, n2, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, nBytesTraces, nBytesHdrs);
    }

    private static int encodeHdr(byte[] encodedData, short cookie, float distortion, float ftGainExponent, int n1, int n2, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength, int nBytesTraces, int nBytesHdrs) {
        int encodedDataIndex = 0;
        BlockCompressor.stuffShortInBytes(cookie, encodedData, encodedDataIndex);
        int idistortion = Float.floatToIntBits(distortion);
        BlockCompressor.stuffIntInBytes(idistortion, encodedData, encodedDataIndex += 2);
        BlockCompressor.stuffIntInBytes(n1, encodedData, encodedDataIndex += 4);
        BlockCompressor.stuffIntInBytes(n2, encodedData, encodedDataIndex += 4);
        encodedDataIndex += 4;
        if (verticalBlockSize > Short.MAX_VALUE) {
            throw new RuntimeException("verticalBlockSize > Short.MAX_VALUE");
        }
        BlockCompressor.stuffShortInBytes((short)verticalBlockSize, encodedData, encodedDataIndex);
        encodedDataIndex += 2;
        if (horizontalBlockSize > Short.MAX_VALUE) {
            throw new RuntimeException("horizontalBlockSize > Short.MAX_VALUE");
        }
        BlockCompressor.stuffShortInBytes((short)horizontalBlockSize, encodedData, encodedDataIndex);
        encodedDataIndex += 2;
        if (verticalTransLength > 127) {
            throw new RuntimeException("verticalTransLength > Byte.MAX_VALUE");
        }
        encodedData[encodedDataIndex] = (byte)verticalTransLength;
        ++encodedDataIndex;
        if (horizontalTransLength > 127) {
            throw new RuntimeException("horizontalTransLength > Byte.MAX_VALUE");
        }
        encodedData[encodedDataIndex] = (byte)horizontalTransLength;
        BlockCompressor.stuffIntInBytes(nBytesTraces, encodedData, ++encodedDataIndex);
        BlockCompressor.stuffIntInBytes(nBytesHdrs, encodedData, encodedDataIndex += 4);
        encodedDataIndex += 4;
        if (cookie == 30744 || cookie == 29941) {
            int iftGainExponent = Float.floatToIntBits(ftGainExponent);
            BlockCompressor.stuffIntInBytes(iftGainExponent, encodedData, encodedDataIndex);
            encodedDataIndex += 4;
        }
        return encodedDataIndex;
    }

    static void checkDataIntegrity(byte[] encodedData) {
        int[] hdrInfo = new int[11];
        SeisPEG.decodeHdr(encodedData, hdrInfo);
    }

    private static boolean badAmplitudeData(byte[] encodedData) {
        int cookie = BlockCompressor.stuffBytesInShort(encodedData, 0);
        if (cookie == 30607 || cookie == 30744) {
            return false;
        }
        if (cookie == 29899 || cookie == 29941) {
            return true;
        }
        throw new RuntimeException("Sorry - you are trying to uncompress data from an unsupported unreleased version of SeisPEG (contact ddiller@tierrageo.com if this is unacceptable) " + cookie + " " + 30607 + " " + 30744);
    }

    private static int decodeHdr(byte[] encodedData, int[] hdrInfo) {
        int encodedDataIndex = 0;
        int cookie = BlockCompressor.stuffBytesInShort(encodedData, encodedDataIndex);
        if (cookie != 30607 && cookie != 29899 && cookie != 30744 && cookie != 29941) {
            throw new RuntimeException("Sorry - you are trying to uncompress data from an unsupported unreleased version of SeisPEG (contact ddiller@tierrageo.com if this is unacceptable) " + cookie + " " + 30607 + " " + 30744);
        }
        int idistortion = BlockCompressor.stuffBytesInInt(encodedData, encodedDataIndex += 2);
        int n1 = BlockCompressor.stuffBytesInInt(encodedData, encodedDataIndex += 4);
        int n2 = BlockCompressor.stuffBytesInInt(encodedData, encodedDataIndex += 4);
        int verticalBlockSize = BlockCompressor.stuffBytesInShort(encodedData, encodedDataIndex += 4);
        int horizontalBlockSize = BlockCompressor.stuffBytesInShort(encodedData, encodedDataIndex += 2);
        int verticalTransLength = encodedData[encodedDataIndex += 2];
        int horizontalTransLength = encodedData[++encodedDataIndex];
        int nBytesTraces = BlockCompressor.stuffBytesInInt(encodedData, ++encodedDataIndex);
        int nBytesHdrs = BlockCompressor.stuffBytesInInt(encodedData, encodedDataIndex += 4);
        encodedDataIndex += 4;
        int iftGainExponent = 0;
        if (cookie == 30744 || cookie == 29941) {
            iftGainExponent = BlockCompressor.stuffBytesInInt(encodedData, encodedDataIndex);
            encodedDataIndex += 4;
        }
        hdrInfo[0] = cookie;
        hdrInfo[1] = idistortion;
        hdrInfo[2] = n1;
        hdrInfo[3] = n2;
        hdrInfo[4] = verticalBlockSize;
        hdrInfo[5] = horizontalBlockSize;
        hdrInfo[6] = verticalTransLength;
        hdrInfo[7] = horizontalTransLength;
        hdrInfo[8] = nBytesTraces;
        hdrInfo[9] = nBytesHdrs;
        hdrInfo[10] = iftGainExponent;
        for (int i = 2; i < 7; ++i) {
            if (hdrInfo[i] >= 1) continue;
            throw new RuntimeException("Invalid header - data corrupted?");
        }
        return encodedDataIndex;
    }

    int compress2D(float[] paddedTraces, int n1, int n2, int paddedN1, int paddedN2, float distortion, float ftGainExponent, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength, byte[] encodedData, int outputBufferSize) {
        if (paddedN1 == 0 || paddedN2 == 0 || verticalBlockSize == 0 || horizontalBlockSize == 0 || verticalTransLength == 0 || horizontalBlockSize == 0) {
            throw new IllegalArgumentException("Invalid args");
        }
        this._n1 = n1;
        this._n2 = n2;
        this._paddedN1 = paddedN1;
        this._paddedN2 = paddedN2;
        this._verticalBlockSize = verticalBlockSize;
        this._horizontalBlockSize = horizontalBlockSize;
        this._verticalTransLength = verticalTransLength;
        this._horizontalTransLength = horizontalTransLength;
        return this.compress2D(paddedTraces, distortion, ftGainExponent, encodedData, outputBufferSize);
    }

    private float[] computeFtGain(int samplesPerTrace, float ftGainExponent) {
        float[] ftGain = new float[samplesPerTrace];
        double sum = 0.0;
        for (int i = 0; i < ftGain.length; ++i) {
            double time = (double)i * 4.0 / 1000.0;
            ftGain[i] = (float)Math.pow(time, ftGainExponent) + 0.001f;
            sum += (double)ftGain[i];
        }
        float averageGain = (float)(sum / (double)ftGain.length);
        int i = 0;
        while (i < ftGain.length) {
            int n = i++;
            ftGain[n] = ftGain[n] / averageGain;
        }
        return ftGain;
    }

    private static void applyFtGain(float[] ftGain, float[] buffer, int index, int n1) {
        for (int i = 0; i < n1; ++i) {
            int n = index + i;
            buffer[n] = buffer[n] * ftGain[i];
        }
    }

    private static void removeFtGain(float[] ftGain, float[] trace) {
        for (int i = 0; i < trace.length; ++i) {
            int n = i;
            trace[n] = trace[n] / ftGain[i];
        }
    }

    private int compress2D(float[] paddedTraces, float distortion, float ftGainExponent, byte[] encodedData, int outputBufferSize) {
        this.x1Transform(1, paddedTraces);
        this.timeTransform(1, paddedTraces);
        short cookie = (double)ftGainExponent == 0.0 ? (short)30607 : 30744;
        int nbytesHdr = SeisPEG.encodeHdr(encodedData, cookie, distortion, ftGainExponent, this._n1, this._n2, this._verticalBlockSize, this._horizontalBlockSize, this._verticalTransLength, this._horizontalTransLength, 0, 0);
        int nbytesData = this.codeAllBlocks(1, paddedTraces, this._paddedN1, this._paddedN2, distortion, this._verticalBlockSize, this._horizontalBlockSize, encodedData, nbytesHdr, outputBufferSize - nbytesHdr);
        if (nbytesData == 0) {
            return 0;
        }
        int nBytesTotal = nbytesHdr + nbytesData;
        SeisPEG.decodeHdr(encodedData, this._hdrInfo);
        this._hdrInfo[8] = nBytesTotal;
        this._hdrInfo[9] = 0;
        SeisPEG.updateHdr(encodedData, this._hdrInfo);
        return nBytesTotal;
    }

    int uncompress2D(byte[] encodedData, int inputBufferSize, float[] paddedTraces) {
        int nbytesHdr = SeisPEG.decodeHdr(encodedData, this._hdrInfo);
        this._n1 = this._hdrInfo[2];
        this._n2 = this._hdrInfo[3];
        this._verticalBlockSize = this._hdrInfo[4];
        this._horizontalBlockSize = this._hdrInfo[5];
        this._verticalTransLength = this._hdrInfo[6];
        this._horizontalTransLength = this._hdrInfo[7];
        this._paddedN1 = SeisPEG.computePaddedLength(this._n1, this._verticalBlockSize);
        this._paddedN2 = SeisPEG.computePaddedLength(this._n2, this._horizontalBlockSize);
        float distortion = 0.0f;
        int nbytes = this.codeAllBlocks(-1, paddedTraces, this._paddedN1, this._paddedN2, distortion, this._verticalBlockSize, this._horizontalBlockSize, encodedData, nbytesHdr, inputBufferSize - nbytesHdr);
        if (nbytes == 0) {
            return -1;
        }
        this.timeTransform(-1, paddedTraces);
        this.x1Transform(-1, paddedTraces);
        return 0;
    }

    public static int getOutputHdrBufferSize(int maxHdrLength, int maxTracesPerFrame) {
        return HdrCompressor.getOutputBufferSize(maxHdrLength, maxTracesPerFrame);
    }

    public int compressHdrs(int[][] hdrs, int hdrLength, int nTraces, byte[] encodedBytes) {
        return this._hdrCompressor.compress(hdrs, hdrLength, (float[][])null, nTraces, encodedBytes, 0);
    }

    public int compress(float[][] traces, int nTraces, IntBuffer hdrIntBuffer, int hdrLength, byte[] outputData) {
        int nBytesTraces = this.compress(traces, nTraces, outputData);
        int nBytesHdrs = this._hdrCompressor.compress(hdrIntBuffer, hdrLength, traces, nTraces, outputData, nBytesTraces);
        SeisPEG.decodeHdr(outputData, this._hdrInfo);
        this._hdrInfo[8] = nBytesTraces;
        this._hdrInfo[9] = nBytesHdrs;
        SeisPEG.updateHdr(outputData, this._hdrInfo);
        return nBytesTraces + nBytesHdrs;
    }

    public int uncompressHdrs(byte[] encodedBytes, int nBytes, int[][] hdrs) throws DataCorruptedException, DataFormatException {
        return this._hdrCompressor.uncompress(encodedBytes, 0, nBytes, hdrs, (float[][])null);
    }

    public int uncompress(byte[] compressedByteData, int compressedDataLength, float[][] traces, IntBuffer hdrIntBuffer) throws DataCorruptedException, DataFormatException {
        SeisPEG.decodeHdr(compressedByteData, this._hdrInfo);
        this.uncompress(compressedByteData, compressedDataLength, traces);
        int nBytesTraces = this._hdrInfo[8];
        int nBytesHdrs = this._hdrInfo[9];
        return this._hdrCompressor.uncompress(compressedByteData, nBytesTraces, nBytesHdrs, hdrIntBuffer, traces);
    }

    public void updateStatistics(int nTracesWritten, int traceLength, int hdrLength, int nBytes) {
        this._traceLength = traceLength;
        this._hdrLength = hdrLength;
        this._nTracesWrittenTotal += (long)nTracesWritten;
        this._nBytesTotal += (long)nBytes;
    }

    public long countTracesWritten() {
        return this._nTracesWrittenTotal;
    }

    public double getCompressionRatio() {
        return (double)((long)this._traceLength * this._nTracesWrittenTotal * 4L + (long)this._hdrLength * this._nTracesWrittenTotal * 4L) / (double)this._nBytesTotal;
    }

    static {
        String s = System.getProperty("org.javaseis.seiszip.nthreads");
        if (s != null) {
            try {
                c_nThreads = Integer.parseInt(s);
            }
            catch (Exception e) {
                c_nThreads = 2;
                LOG.warning("Unable to parse 'org.javaseis.seiszip.nthreads' property '" + s + "'");
            }
        }
        LOG.info("SeisPEG is using " + c_nThreads + " threads - reset with org.javaseis.seiszip.nthreads=n");
    }

    private class X1Transformer
    extends Thread {
        private int _myThreadIndex;
        private int _nThreads;
        private Transformer _myTransformer = new Transformer();
        private int _direction;
        private float[] _paddedTraces = null;
        private float[] _myScratch2 = null;
        private float[][] _myVecW = null;

        public X1Transformer(int threadIndex, int nThreads) {
            this._myThreadIndex = threadIndex;
            this._nThreads = nThreads;
        }

        public synchronized void setData(int direction, float[] paddedTraces) {
            this._direction = direction;
            this._paddedTraces = paddedTraces;
            this.notifyAll();
        }

        public synchronized void waitForFinish() {
            while (this._paddedTraces != null) {
                try {
                    this.wait();
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    continue;
                }
                break;
            }
            return;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (true) {
                X1Transformer x1Transformer = this;
                synchronized (x1Transformer) {
                    if (this._paddedTraces != null) {
                        int nblocksHorizontal = SeisPEG.this._paddedN2 / SeisPEG.this._horizontalBlockSize;
                        assert (nblocksHorizontal * SeisPEG.this._horizontalBlockSize == SeisPEG.this._paddedN2);
                        if (this._myVecW == null) {
                            this._myVecW = new float[32][SeisPEG.this._paddedN2];
                        }
                        if (this._myScratch2 == null) {
                            this._myScratch2 = new float[SeisPEG.this._paddedN2 + SeisPEG.this._horizontalBlockSize];
                        }
                        int nsamps = SeisPEG.this._paddedN1;
                        for (int i = this._myThreadIndex * 32; i < nsamps; i += 32 * this._nThreads) {
                            int l;
                            int l2;
                            int index;
                            int m;
                            int n = nsamps - i;
                            if (n > 32) {
                                n = 32;
                            }
                            for (m = 0; m < SeisPEG.this._paddedN2; ++m) {
                                index = m * SeisPEG.this._paddedN1 + i;
                                for (l2 = 0; l2 < n; ++l2) {
                                    this._myVecW[l2][m] = this._paddedTraces[l2 + index];
                                }
                            }
                            if (this._direction == 1) {
                                for (l = 0; l < n; ++l) {
                                    this._myTransformer.lotFwd(this._myVecW[l], 0, SeisPEG.this._horizontalBlockSize, SeisPEG.this._horizontalTransLength, nblocksHorizontal, this._myScratch2);
                                }
                            } else {
                                for (l = 0; l < n; ++l) {
                                    this._myTransformer.lotRev(this._myVecW[l], 0, SeisPEG.this._horizontalBlockSize, SeisPEG.this._horizontalTransLength, nblocksHorizontal, this._myScratch2);
                                }
                            }
                            for (m = 0; m < SeisPEG.this._paddedN2; ++m) {
                                index = m * SeisPEG.this._paddedN1 + i;
                                for (l2 = 0; l2 < n; ++l2) {
                                    this._paddedTraces[l2 + index] = this._myVecW[l2][m];
                                }
                            }
                        }
                        this._paddedTraces = null;
                        this.notifyAll();
                    } else {
                        try {
                            this.wait();
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                    }
                }
            }
        }
    }

    private class TimeTransformer
    extends Thread {
        private int _myThreadIndex;
        private int _nThreads;
        private Transformer _myTransformer = new Transformer();
        private int _direction;
        private float[] _paddedTraces = null;
        private float[] _myScratch1 = null;

        public TimeTransformer(int threadIndex, int nThreads) {
            this._myThreadIndex = threadIndex;
            this._nThreads = nThreads;
        }

        public synchronized void setData(int direction, float[] paddedTraces) {
            this._direction = direction;
            this._paddedTraces = paddedTraces;
            this.notifyAll();
        }

        public synchronized void waitForFinish() {
            while (this._paddedTraces != null) {
                try {
                    this.wait();
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    continue;
                }
                break;
            }
            return;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (true) {
                TimeTransformer timeTransformer = this;
                synchronized (timeTransformer) {
                    if (this._paddedTraces != null) {
                        int nblocksVertical = SeisPEG.this._paddedN1 / SeisPEG.this._verticalBlockSize;
                        assert (nblocksVertical * SeisPEG.this._verticalBlockSize == SeisPEG.this._paddedN1);
                        if (this._myScratch1 == null) {
                            this._myScratch1 = new float[SeisPEG.this._paddedN1 + SeisPEG.this._verticalBlockSize];
                        }
                        for (int j = this._myThreadIndex; j < SeisPEG.this._paddedN2; j += this._nThreads) {
                            int index = j * SeisPEG.this._paddedN1;
                            if (this._direction == 1) {
                                this._myTransformer.lotFwd(this._paddedTraces, index, SeisPEG.this._verticalBlockSize, SeisPEG.this._verticalTransLength, nblocksVertical, this._myScratch1);
                                continue;
                            }
                            this._myTransformer.lotRev(this._paddedTraces, index, SeisPEG.this._verticalBlockSize, SeisPEG.this._verticalTransLength, nblocksVertical, this._myScratch1);
                        }
                        this._paddedTraces = null;
                        this.notifyAll();
                    } else {
                        try {
                            this.wait();
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                    }
                }
            }
        }
    }

    public static enum Policy {
        FASTEST,
        MAX_COMPRESSION;

    }
}

