#include "jmwrapper.h"
#include "javavm.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

static jmethodID NONE = 0;
static jmethodID BYTES = 0;
static jmethodID FLOATS = 0;
static jmethodID DOUBLES = 0;
static jmethodID INTS = 0;
static jmethodID LONGS = 0;

/* prototypes for some private convenience functions */

int jmwrapper_alloc_byte_array (JMWrapper *jmw,
  size_t size); /* size of byte array to allocate */ 

int jmwrapper_alloc_float_array (JMWrapper *jmw,
  size_t size); /* size of float array to allocate */ 

int jmwrapper_alloc_double_array (JMWrapper *jmw,
  size_t size); /* size of double array to allocate */ 

int jmwrapper_alloc_int_array (JMWrapper *jmw,
  size_t size); /* size of int array to allocate */ 

int jmwrapper_alloc_long_array (JMWrapper *jmw,
  size_t size); /* size of long array to allocate */ 

void jmwrapper_copy_char_to_jbyte (
  char *chrs,    /* from array */
  jbyte *jbyts,  /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_jbyte_to_char (
  jbyte *jbyts,  /* from array */
  char *chrs,    /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_float_to_jfloat (
  float *flts,   /* from array */
  jfloat *jflts, /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_jfloat_to_float (
  jfloat *jflts, /* from array */
  float *flts,   /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_double_to_jdouble (
  double *dbls,   /* from array */
  jdouble *jdbls, /* to array */
  size_t count);  /* size of arrays */

void jmwrapper_copy_jdouble_to_double (
  jdouble *jdbls, /* from array */
  double *dbls,   /* to array */
  size_t count);  /* size of arrays */

void jmwrapper_copy_int_to_jint (
  int *ints,     /* from array */
  jint *jints,   /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_jint_to_int (
  jint *jints,   /* from array */
  int *ints,     /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_long_to_jlong (
  long *lngs,    /* from array */
  jlong *jlngs,  /* to array */
  size_t count); /* size of arrays */

void jmwrapper_copy_jlong_to_long (
  jlong* jlngs,  /* from array */
  long* lngs,    /* to array */
  size_t count); /* size of arrays */

/* limited to a java constructor having no arguments */
JMWrapper *jmwrapper_create (const char *class_name)
{
  JMWrapper *jmw = NULL;

  JNIEnv *env;

  jclass loc_cls;
  jobject loc_obj;
  jmethodID mid;

  /* allocate the memory for the JMWrapper */
  jmw = (JMWrapper *)malloc (sizeof(JMWrapper));
  if (jmw == NULL) return jmw;

  env = (javavm_fetch())->_env;

  /* find the class */
  loc_cls = (*env)->FindClass (env, class_name);
  if (loc_cls == 0) {
    if ((*env)->ExceptionOccurred(env)) {
      (*env)->ExceptionDescribe (env);
    }
    fprintf (stderr, "Failed to find %s class\n", class_name);
    return NULL;
  }
  
  /* prevent access to class from being garbage collected */
  jmw->_cls = (*env)->NewWeakGlobalRef (env, loc_cls);
  (*env)->DeleteLocalRef (env, loc_cls);
  if (jmw->_cls == NULL) {
    fprintf (stderr, "Failed to allocate memory for access to %s\n",
      class_name);
    return NULL;
  }

  /* find the constructor */
  mid = (*env)->GetMethodID (env, jmw->_cls, "<init>",
    "()V");
  if (mid == 0) {
    if ((*env)->ExceptionOccurred(env)) {
      (*env)->ExceptionDescribe (env);
    }
    fprintf (stderr, "Failed to find %s constructor\n", class_name);
    return NULL;
  }

  /* instantiate an object */
  loc_obj = (*env)->NewObject (env, jmw->_cls, mid);
  if (loc_obj == NULL) {
    fprintf (stderr, "Failed to instantiate a %s object\n", class_name);
    return NULL;
  }

  /* prevent object instance from being gargabe collected */
  jmw->_obj = (*env)->NewGlobalRef (env, loc_obj);
  (*env)->DeleteLocalRef (env, loc_obj);
  if (jmw->_obj == NULL) {
    fprintf (stderr, "Failed to allocate memory for instance of %s\n",
      class_name);
    return NULL;
  }

  /* initialize each type of array */
  jmw->_jarray_byts = NULL;
  jmw->_byts = NULL;
  jmw->_byts_size = 0;
  jmw->_jarray_flts = NULL;
  jmw->_flts = NULL;
  jmw->_flts_size = 0;
  jmw->_jarray_dbls = NULL;
  jmw->_dbls = NULL;
  jmw->_dbls_size = 0;
  jmw->_jarray_ints = NULL;
  jmw->_ints = NULL;
  jmw->_ints_size = 0;
  jmw->_jarray_lngs = NULL;
  jmw->_lngs = NULL;
  jmw->_lngs_size = 0;
  return jmw;
}

/* limited to a Java method using 1 args: in Java String with an int * returned */
int jmwrapper_none (JMWrapper *jmw, char *desc)
{
  int retval = 0;
  jstring jdesc;

  jint jretval;
  JNIEnv *env = (javavm_fetch())->_env;

  /* find the method */ 
  if (NONE == 0) {
    NONE = (*env)->GetMethodID (env, jmw->_cls, "none",
      "(Ljava/lang/String;)I");
    if (NONE == 0) {
      fprintf (stderr, "Failed to find none method\n");
      return retval;
    }
  }

  /* create a local Java string to contain the given description */
  jdesc = (*env)->NewStringUTF (env, desc);
  if (jdesc == 0) {
    fprintf (stderr, "Out of memory\n");
    return retval;
  }

  /* call the method */
  jretval = (*env)->CallIntMethod (env, jmw->_obj, NONE, jdesc);

  retval = (int)jretval;
  return retval;
}

/* limited to a Java method using 2 args: in Java String and an in/out byte array with an int * returned */
int jmwrapper_chars (JMWrapper *jmw, char *desc, char *chrs,
  size_t max_size, int io)
{
  int retval = 0;
  jstring jdesc;

  if (max_size < 1) return retval;

  jint jcount;
  JNIEnv *env = (javavm_fetch())->_env;

  /* find the method */
  if (BYTES == 0) {
    BYTES = (*env)->GetMethodID (env, jmw->_cls, "bytes",
      "(Ljava/lang/String;[B)I");
    if (BYTES == 0) {
      fprintf (stderr, "Failed to find bytes method\n");
      return retval;
    }
  }

  /* create a local Java string to contain the given description */
  jdesc = (*env)->NewStringUTF (env, desc);
  if (jdesc == 0) {
    fprintf (stderr, "Out of memory\n");
    return retval;
  }

  /* allocate a local Java byte array to contain the C jbyte array */
  if (jmwrapper_alloc_byte_array(jmw,max_size) == 0) {
    return retval;
  }

  if (io == JMW_IN || io == JMW_BOTH) {
    /* convert the incoming char array to a C jbyte array */
    jmwrapper_copy_char_to_jbyte (chrs, jmw->_byts, max_size);
    /* convert the C jbyte array to a Java byte array */
    (*env)->SetByteArrayRegion (env, jmw->_jarray_byts, (jsize)0,
      (jsize)max_size, jmw->_byts);
  }

  /* call the method to pass the Java byte array for input/output */
  jcount = (*env)->CallIntMethod (env, jmw->_obj, BYTES, jdesc,
    jmw->_jarray_byts);
  assert (jcount <= max_size); /* fatal error */

  if ((io == JMW_OUT || io == JMW_BOTH) && jcount > 0) {
    if (!(jcount < max_size)) {
      assert (0); /* fatal error on output */
    }
    /* convert the outgoing Java byte array to a C jbyte array */
    (*env)->GetByteArrayRegion (env, jmw->_jarray_byts, (jsize)0,
      (jsize)jcount, jmw->_byts);
    /* convert the C jbyte array to a char array */
    jmwrapper_copy_jbyte_to_char (jmw->_byts, chrs, (size_t)jcount);
  }
  retval = jcount > 0 ? 1 : 0;
  return retval;
}

/* limited to a Java method using 3 args: in/out float array, size, and io with an int returned */
int jmwrapper_floats (JMWrapper *jmw, char *desc, float *flts,
  size_t size, int io)
{
  int retval = 0;
  jstring jdesc;
  
  JNIEnv *env = (javavm_fetch())->_env;
  jint jretval;

  /* find the method */ 
  if (FLOATS == 0) {
    FLOATS = (*env)->GetMethodID (env, jmw->_cls, "floats",
      "(Ljava/lang/String;[F)I");
    if (FLOATS == 0) {
      fprintf (stderr, "Failed to find floats method\n");
      return retval;
    }
  }

  /* create a local Java string to contain the given description */
  jdesc = (*env)->NewStringUTF (env, desc);
  if (jdesc == 0) {
    fprintf (stderr, "Out of memory\n");
    return retval;
  }

  /* allocate a local Java float array to contain the float array */
  if (jmwrapper_alloc_float_array(jmw,size) == 0) {
    return retval;
  }

  if (io == JMW_IN || io == JMW_BOTH) {
    /* convert the incoming float array to a C jfloat array */
    jmwrapper_copy_float_to_jfloat (flts, jmw->_flts, size);
    /* convert the C jfloat array to a Java float array */
    (*env)->SetFloatArrayRegion (env, jmw->_jarray_flts, (jsize)0,
      (jsize)size, jmw->_flts);
  }

  /* call the method to populate the local java float array */
  jretval = (*env)->CallIntMethod (env, jmw->_obj, FLOATS, jdesc,
    jmw->_jarray_flts);
  retval = (int)jretval;

  if ((io == JMW_OUT || io == JMW_BOTH) && retval > 0) {
    /* convert the outgoing Java float array to a C jfloat array */
    (*env)->GetFloatArrayRegion (env, jmw->_jarray_flts, (jsize)0,
      (jsize)size, jmw->_flts);
    /* convert the C jfloat array to a float array */
    jmwrapper_copy_jfloat_to_float (jmw->_flts, flts,
      size);
  }
  return retval;
}

/* limited to a Java method using 3 args: in/out double array, size, and io with an int returned */
int jmwrapper_doubles (JMWrapper *jmw, char *desc, double *dbls,
  size_t size, int io)
{
  int retval = 0;
  jstring jdesc;
  
  JNIEnv *env = (javavm_fetch())->_env;
  jint jretval; 

  /* find the method */
  if (DOUBLES == 0) {
    DOUBLES = (*env)->GetMethodID (env, jmw->_cls, "doubles",
      "(Ljava/lang/String;[D)I");
    if (DOUBLES == 0) {
      fprintf (stderr, "Failed to find doubles method\n");
      return retval;
    }
  }

  /* create a local Java string to contain the given description */
  jdesc = (*env)->NewStringUTF (env, desc);
  if (jdesc == 0) {
    fprintf (stderr, "Out of memory\n");
    return retval;
  }

  /* allocate a local Java double array to contain the double array */
  if (jmwrapper_alloc_double_array(jmw,size) == 0) {
    return retval;
  }

  if (io == JMW_IN || io == JMW_BOTH) {
    /* convert the incoming double array to a C jdouble array */
    jmwrapper_copy_double_to_jdouble (dbls, jmw->_dbls, size);
    /* convert the C jdouble array to a Java double array */
    (*env)->SetDoubleArrayRegion (env, jmw->_jarray_dbls, (jsize)0,
      (jsize)size, jmw->_dbls);
  }

  /* call the method to populate the local java double array */
  jretval = (*env)->CallIntMethod (env, jmw->_obj, DOUBLES, jdesc,
    jmw->_jarray_dbls);
  retval = (int)jretval;

  if ((io == JMW_OUT || io == JMW_BOTH) && retval > 0) {
    /* convert the outgoing Java double array to a C jdouble array */
    (*env)->GetDoubleArrayRegion (env, jmw->_jarray_dbls, (jsize)0,
      (jsize)size, jmw->_dbls);
    /* convert the C jdouble array to a double array */
    jmwrapper_copy_jdouble_to_double (jmw->_dbls, dbls,
      size);
  }
  return retval;
}

/* limited to a Java method using 3 args: in/out int array, size, and io with an int returned */
int jmwrapper_ints (JMWrapper *jmw, char *desc, int *ints,
  size_t size, int io)
{
  int retval = 0;
  jstring jdesc;
  
  JNIEnv *env = (javavm_fetch())->_env;
  jint jretval; 

  /* find the method */
  if (INTS == 0) {
   INTS = (*env)->GetMethodID (env, jmw->_cls, "ints",
     "(Ljava/lang/String;[I)I");
    if (INTS == 0) {
      fprintf (stderr, "Failed to find ints method\n");
      return retval;
    }
  }

  /* create a local Java string to contain the given description */
  jdesc = (*env)->NewStringUTF (env, desc);
  if (jdesc == 0) {
    fprintf (stderr, "Out of memory\n");
    return retval;
  }

  /* allocate a local Java float array to contain the int array */
  if (jmwrapper_alloc_int_array(jmw,size) == 0) {
    return retval;
  }

  if (io == JMW_IN || io == JMW_BOTH) {
    /* convert the incoming int array to a C jint array */
    jmwrapper_copy_int_to_jint (ints, jmw->_ints, size);
    /* convert the C jint array to a Java int array */
    (*env)->SetIntArrayRegion (env, jmw->_jarray_ints, (jsize)0,
      (jsize)size, jmw->_ints);
  }

  /* call the method to populate the local java int array */
  jretval = (*env)->CallIntMethod (env, jmw->_obj, INTS, jdesc,
    jmw->_jarray_ints);
  retval = (int)jretval;

  if ((io == JMW_OUT || io == JMW_BOTH) && retval > 0) {
    /* convert the outgoing Java int array to a C jint array */
    (*env)->GetIntArrayRegion (env, jmw->_jarray_ints, (jsize)0,
      (jsize)size, jmw->_ints);
    /* convert the C jint array to a int array */
    jmwrapper_copy_jint_to_int (jmw->_ints, ints,
      size);
  }
  return retval;
}

/* limited to a Java method using 3 args: in/out long array, size, and io with an int returned */
int jmwrapper_longs (JMWrapper *jmw, char *desc, long *lngs,
  size_t size, int io)
{
  int retval = 0;
  jstring jdesc;
  
  JNIEnv *env = (javavm_fetch())->_env;
  jint jretval; 

  /* find the method */
  if (LONGS == 0) {
    LONGS = (*env)->GetMethodID (env, jmw->_cls, "longs",
      "(Ljava/lang/String;[J)I");
    if (LONGS == 0) {
      fprintf (stderr, "Failed to find longs method\n");
      return retval;
    }
  }

  /* create a local Java string to contain the given description */
  jdesc = (*env)->NewStringUTF (env, desc);
  if (jdesc == 0) {
    fprintf (stderr, "Out of memory\n");
    return retval;
  }

  /* allocate a local Java long array to contain the long array */
  if (jmwrapper_alloc_long_array(jmw,size) == 0) {
    return retval;
  }

  if (io == JMW_IN || io == JMW_BOTH) {
    /* convert the incoming long array to a C jlong array */
    jmwrapper_copy_long_to_jlong (lngs, jmw->_lngs, size);
    /* convert the C jlong array to a Java long array */
    (*env)->SetLongArrayRegion (env, jmw->_jarray_lngs, (jsize)0,
      (jsize)size, jmw->_lngs);
  }

  /* call the method to populate the local java long array */
  jretval = (*env)->CallIntMethod (env, jmw->_obj, LONGS, jdesc,
    jmw->_jarray_lngs);
  retval = (int)jretval;

  if ((io == JMW_OUT || io == JMW_BOTH) && retval > 0) {
    /* convert the outgoing Java long array to a C jlong array */
    (*env)->GetLongArrayRegion (env, jmw->_jarray_lngs, (jsize)0,
      (jsize)size, jmw->_lngs);
    /* convert the C jlong array to a long array */
    jmwrapper_copy_jlong_to_long (jmw->_lngs, lngs, size);
  }
  return retval;
}

void jmwrapper_delete (JMWrapper *jmw)
{
  JNIEnv *env = (javavm_fetch())->_env;

  if (jmw != NULL) {
    (*env)->DeleteWeakGlobalRef (env, jmw->_cls);
    (*env)->DeleteGlobalRef (env, jmw->_obj);
    if (jmw->_byts != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_byts);
      jmw->_jarray_byts = NULL;
      free (jmw->_byts);
      jmw->_byts = NULL;
      jmw->_byts_size = 0;
    }
    if (jmw->_flts != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_flts);
      jmw->_jarray_flts = NULL;
      free (jmw->_flts);
      jmw->_flts = NULL;
      jmw->_flts_size = 0;
    }
    if (jmw->_dbls != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_dbls);
      jmw->_jarray_dbls = NULL;
      free (jmw->_dbls);
      jmw->_dbls = NULL;
      jmw->_dbls_size = 0;
    }
    if (jmw->_ints != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_ints);
      jmw->_jarray_ints = NULL;
      free (jmw->_ints);
      jmw->_ints = NULL;
      jmw->_ints_size = 0;
    }
    if (jmw->_lngs != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_lngs);
      jmw->_jarray_lngs = NULL;
      free (jmw->_lngs);
      jmw->_lngs = NULL;
      jmw->_lngs_size = 0;
    }
    free (jmw);
  }
}

int jmwrapper_alloc_byte_array (JMWrapper *jmw, size_t size)
{
  int retval;

  jbyte *tmp;
  JNIEnv *env = (javavm_fetch())->_env;
  jarray loc_jarray_byts;

  /* 
   * in the past an attemp was made to use a high-water mark concept
   *   this did not play nice with Java thus jmwrapper insists that the
   *   array sizes are exact matches
   */
  if (size != jmw->_byts_size) {

    if (jmw->_jarray_byts != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_byts);
      jmw->_jarray_byts = NULL;
    }

    /* create the Java byte array */
    loc_jarray_byts = (*env)->NewByteArray (env, (jsize)size);
    if (loc_jarray_byts == 0) {
      fprintf (stderr, "Failed to allocate the Java byte array\n");
      retval = 0;
    }
    else {
      jmw->_jarray_byts = (*env)->NewGlobalRef (env, loc_jarray_byts);
      (*env)->DeleteLocalRef (env, loc_jarray_byts);
      if (jmw->_byts_size != 0) {
        free (jmw->_byts);
        jmw->_byts = NULL;
        jmw->_byts_size = 0;
      }
      /* create the C jbyte array */
      tmp = (jbyte *)malloc (size*sizeof(jbyte));
      if (tmp != NULL) {
        /* successfully allocated the array */
        jmw->_byts = tmp;
        jmw->_byts_size  = (jsize)size;
        retval = 1;
      }
      else {
        fprintf (stderr, "Failed to allocate the jbyte array\n");
        retval = 0;
      }
    }
  }
  else {
    retval = 1;
  }
  return retval;
}

int jmwrapper_alloc_float_array (JMWrapper *jmw, size_t size)
{
  int retval;

  jfloat *tmp;
  JNIEnv *env = (javavm_fetch())->_env;
  jarray loc_jarray_flts;

  /* 
   * in the past an attemp was made to use a high-water mark concept
   *   this did not play nice with Java thus jmwrapper insists that the
   *   array sizes are exact matches
   */
  if (size != jmw->_flts_size) {

    if (jmw->_jarray_flts != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_flts);
      jmw->_jarray_flts = NULL;
    }

    /* create the Java float array */
    loc_jarray_flts = (*env)->NewFloatArray (env, (jsize)size);
    if (loc_jarray_flts == 0) {
      fprintf (stderr, "Failed to allocate the Java float array\n");
      retval = 0;
    }
    else {
      jmw->_jarray_flts = (*env)->NewGlobalRef (env, loc_jarray_flts);
      (*env)->DeleteLocalRef (env, loc_jarray_flts);
      if (jmw->_flts_size != 0) {
        free (jmw->_flts);
        jmw->_flts = NULL;
        jmw->_flts_size = 0;
      }
      /* create the C jfloat array */
      tmp = (jfloat *)malloc (size*sizeof(jfloat));
      if (tmp != NULL) {
        /* successfully allocated the array */
        jmw->_flts = tmp;
        jmw->_flts_size  = (jsize)size;
        retval = 1;
      }
      else {
        fprintf (stderr, "Failed to allocate the jfloat array\n");
        retval = 0;
      }
    }
  }
  else {
    retval = 1;
  }
  return retval;
}

int jmwrapper_alloc_double_array (JMWrapper *jmw, size_t size)
{
  int retval;

  jdouble *tmp;
  JNIEnv *env = (javavm_fetch())->_env;
  jarray loc_jarray_dbls;

  /* 
   * in the past an attemp was made to use a high-water mark concept
   *   this did not play nice with Java thus jmwrapper insists that the
   *   array sizes are exact matches
   */
  if (size != jmw->_dbls_size) {

    if (jmw->_jarray_dbls != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_dbls);
      jmw->_jarray_dbls = NULL;
    }

    /* create the Java double array */
    loc_jarray_dbls = (*env)->NewDoubleArray (env, (jsize)size);
    if (loc_jarray_dbls == 0) {
      fprintf (stderr, "Failed to allocate the Java double array\n");
      retval = 0;
    }
    else {
      jmw->_jarray_dbls = (*env)->NewGlobalRef (env, loc_jarray_dbls);
      (*env)->DeleteLocalRef (env, loc_jarray_dbls);
      if (jmw->_dbls_size != 0) {
        free (jmw->_dbls);
        jmw->_dbls = NULL;
        jmw->_dbls_size = 0;
      }
      /* create the C jdouble array */
      tmp = (jdouble *)malloc (size*sizeof(jdouble));
      if (tmp != NULL) {
        /* successfully allocated the array */
        jmw->_dbls = tmp;
        jmw->_dbls_size  = (jsize)size;
        retval = 1;
      }
      else {
        fprintf (stderr, "Failed to allocate the jdouble array\n");
        retval = 0;
      }
    }
  }
  else {
    retval = 1;
  }
  return retval;
}

int jmwrapper_alloc_int_array (JMWrapper *jmw, size_t size)
{
  int retval;

  jint *tmp;
  JNIEnv *env = (javavm_fetch())->_env;
  jarray loc_jarray_ints;

  /* 
   * in the past an attemp was made to use a high-water mark concept
   *   this did not play nice with Java thus jmwrapper insists that the
   *   array sizes are exact matches
   */
  if (size != jmw->_ints_size) {

    if (jmw->_jarray_ints != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_ints);
      jmw->_jarray_ints = NULL;
    }

    /* create the Java int array */
    loc_jarray_ints = (*env)->NewIntArray (env, (jsize)size);
    if (loc_jarray_ints == 0) {
      fprintf (stderr, "Failed to allocate the Java int array\n");
      retval = 0;
    }
    else {
      jmw->_jarray_ints = (*env)->NewGlobalRef (env, loc_jarray_ints);
      (*env)->DeleteLocalRef (env, loc_jarray_ints);
      if (jmw->_ints_size != 0) {
        free (jmw->_ints);
        jmw->_ints = NULL;
        jmw->_ints_size = 0;
      }
      /* create the C jint array */
      tmp = (jint *)malloc (size*sizeof(jint));
      if (tmp != NULL) {
        /* successfully allocated the array */
        jmw->_ints = tmp;
        jmw->_ints_size  = (jsize)size;
        retval = 1;
      }
      else {
        fprintf (stderr, "Failed to allocate the jint array\n");
        retval = 0;
      }
    }
  }
  else {
    retval = 1;
  }
  return retval;
}

int jmwrapper_alloc_long_array (JMWrapper *jmw, size_t size)
{
  int retval;

  jlong *tmp;
  JNIEnv *env = (javavm_fetch())->_env;
  jarray loc_jarray_lngs;

  /* 
   * in the past an attemp was made to use a high-water mark concept
   *   this did not play nice with Java thus jmwrapper insists that the
   *   array sizes are exact matches
   */
  if (size != jmw->_lngs_size) {

    if (jmw->_jarray_lngs != NULL) {
      (*env)->DeleteGlobalRef (env, jmw->_jarray_lngs);
      jmw->_jarray_lngs = NULL;
    }

    /* create the Java long array */
    loc_jarray_lngs = (*env)->NewLongArray (env, (jsize)size);
    if (loc_jarray_lngs == 0) {
      fprintf (stderr, "Failed to allocate the Java long array\n");
      retval = 0;
    }
    else {
      jmw->_jarray_lngs = (*env)->NewGlobalRef (env, loc_jarray_lngs);
      (*env)->DeleteLocalRef (env, loc_jarray_lngs);
      if (jmw->_lngs_size != 0) {
        free (jmw->_lngs);
        jmw->_lngs = NULL;
        jmw->_lngs_size = 0;
      }
      /* create the C jlong array */
      tmp = (jlong *)malloc (size*sizeof(jlong));
      if (tmp != NULL) {
        /* successfully allocated the array */
        jmw->_lngs = tmp;
        jmw->_lngs_size  = (jsize)size;
        retval = 1;
      }
      else {
        fprintf (stderr, "Failed to allocate the jlong array\n");
        retval = 0;
      }
    }
  }
  else {
    retval = 1;
  }
  return retval;
}

void jmwrapper_copy_char_to_jbyte (char *chrs, jbyte *jbyts,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    jbyts[k2] = (jbyte)chrs[k2];
  }
}

/* WARNING: chrs must have a capacity to hold count+1 chars */ 
void jmwrapper_copy_jbyte_to_char (jbyte *jbyts, char *chrs, size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    chrs[k2] = (char)jbyts[k2];
  }
  chrs[count] = '\0'; /* terminate the outgoing char array */
}

void jmwrapper_copy_float_to_jfloat (float *flts, jfloat *jflts,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    jflts[k2] = (jfloat)flts[k2];
  }
}

void jmwrapper_copy_jfloat_to_float (jfloat *jflts, float *flts, size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    flts[k2] = (float)jflts[k2];
  }
}

void jmwrapper_copy_double_to_jdouble (double *dbls, jdouble *jdbls,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    jdbls[k2] = (jdouble)dbls[k2];
  }
}

void jmwrapper_copy_jdouble_to_double (jdouble *jdbls, double *dbls,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    dbls[k2] = (double)jdbls[k2];
  }
}

void jmwrapper_copy_int_to_jint (int *ints, jint *jints,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    jints[k2] = (jint)ints[k2];
  }
}

void jmwrapper_copy_jint_to_int (jint *jints, int *ints,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    ints[k2] = (int)jints[k2];
  }
}

void jmwrapper_copy_long_to_jlong (long *lngs, jlong *jlngs,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    jlngs[k2] = (jlong)lngs[k2];
  }
}

void jmwrapper_copy_jlong_to_long (jlong *jlngs, long *lngs,
  size_t count)
{
  size_t k2;
  for (k2 = 0; k2 < count; k2++) {
    lngs[k2] = (long)jlngs[k2];
  }
}

#ifdef __cplusplus
}
#endif
