#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <sys/types.h>
#ifndef _WIN32
#include <sys/param.h>
#include <sys/utsname.h>
#endif

#ifndef _WIN32
#if (LINUX || LINUXA || LINUXI || LINUXP)
#include <sys/statfs.h>
#else
#include <sys/statvfs.h>
#endif
#else
#include <windows.h>
#include <winioctl.h>
#include <tchar.h>
#include <strsafe.h>
#endif

#define JSUTILFILE_ERROR -7
#define OK 0
#define TRUE 1
#define FALSE 0

int DEBUG = -1;
#ifndef _WIN32
static int o_direct_is_ok (char *filename);
static char *getJS_NFS_O_DIRECT (void);
static int file_is_nfs (char *filename);
static int32_t set_fd_flags (int fd, int32_t flags, int set);
static int fd_flags_set (int fd, int32_t flags);
#endif

int jsutilfile_set_sparse(const char *filename)
{
#ifdef _WIN32
    HANDLE hFile;
    DWORD bytesReturned;
    int retval = 0;
    hFile = CreateFile(TEXT(filename),
		    GENERIC_READ|GENERIC_WRITE,
		    0,
		    NULL,
		    OPEN_ALWAYS,
		    FILE_ATTRIBUTE_NORMAL,
		    NULL);
    if(hFile == INVALID_HANDLE_VALUE) {
	printf("CreateFile(%s) failed error code: %ld\n",filename,(long) GetLastError());
        return retval;
    }

    retval = (0 != DeviceIoControl(
		    hFile,
		    FSCTL_SET_SPARSE,
		    NULL,
		    0,
		    NULL,
		    0,
		    &bytesReturned,
		    NULL));
    (void) CloseHandle(hFile);
    return retval;
#else
    return 1;
#endif
}

/*
 * Set the File Descriptor's O_DIRECT flag if set != 0
 *   otherwise clear flag
 *   return OK (0) on success, otherwise JSUTILFILE_ERROR (-7).
 *   WARNING: this method will return OK if it is determined
 *     that O_DIRECT does not need to be set or cleared! For
 *     example if 1) the file is not on NFS OR 2) if an
 *     JS_NFS_O_DIRECT environment variable is not found AND the
 *     host does not run a Linux 2.6 OS OR 3) JS_NFS_O_DIRECT
 *     is found to be "false". Thus OK may be returned even
 *     when O_DIRECT was not actually set or cleared as requested!
 */
int jsutilfile_set_o_direct (int fd, char *filename, int set)
{
  char* debug = NULL;
  if (DEBUG == -1) {
    debug = getenv ("JS_DEBUG");
    if (debug != NULL) {
      DEBUG = 1;
    }
    else {
      DEBUG = 0;
    }
  }
#ifndef _WIN32

  if (o_direct_is_ok(filename) == TRUE) {
    if (set_fd_flags(fd,O_DIRECT,set) < 0) {
      if (set != 0) {
        printf ("jsutilfile_set_o_direct: WARNING could not SET O_DIRECT flag on %s\n", filename);
      }
      else {
        printf ("jsutilfile_set_o_direct: WARNING could not CLEAR O_DIRECT flag on %s\n", filename);
      }
      return JSUTILFILE_ERROR;
    }
  }
#endif
  return OK;
}

/*
 * return FALSE only if O_DIRECT is relevant and it is
 *   not set, otherwise TRUE
 */
int jsutilfile_o_direct_is_set (int fd, char *filename)
{
  int retval = TRUE;

#ifndef _WIN32
  if (o_direct_is_ok(filename) == TRUE) {
    retval = fd_flags_set (fd, O_DIRECT);
  }
  else {
  	/*
  	 * O_DIRECT cannot be set so assume it is irrelavant
  	 */
  	retval = TRUE;
  }
#endif
  return retval;
}

#ifndef _WIN32
/*
 * return TRUE if it is determined that O_DIRECT is desired or
 *   necessary and the given file is mounted on an NFS.
 */
int o_direct_is_ok (char *filename)
{
  struct utsname u;
  int nfs;
  int retval = FALSE;
  char *js_nfs_o_direct = getJS_NFS_O_DIRECT ();

  if (js_nfs_o_direct != NULL) {
  	/*
  	 * JS_NFS_O_DIRECT environment variable is available
  	 */
  	if (strcmp(js_nfs_o_direct,"FALSE") != 0) {
  	  /*
  	   * JS_NFS_O_DIRECT has not explicitly been set to "false"
  	   */
  	  retval = TRUE;
  	}
  }
  else if (uname(&u) != 0) {
  	/*
  	 * no JS_NFS_O_DIRECT, and cannot execute uname either, so give up
  	 */
  	retval = JSUTILFILE_ERROR;
  }
  else if (strncmp("2.6", u.release, 3) == 0 && strcmp(u.sysname,"Linux") == 0) {
    /*
     * even without JS_NFS_O_DIRECT, we know that direct I/O is required in
     * this situation for nfs files on a Linux kernel 2.6 OS
     */
  	retval = TRUE;
  }

  if (retval == TRUE) {
    nfs = file_is_nfs (filename);
    if (nfs != 1) {
      /*
       * do not bother setting or clearing O_DIRECT for files not on an NFS
       */
      if (DEBUG == 1) printf ("o_direct_is_ok: Not nfs, so setting O_DIRECT is NOT okay\n");
      retval = FALSE;
    }
  }
  return retval;
}

/*
 * returns an uppercase string associated with the JS_NFS_O_DIRECT environment variable
 *         or NULL if it has not been set
 */
char *getJS_NFS_O_DIRECT ()
{
  char *retval = getenv ("JS_NFS_O_DIRECT");
  int k2;

  /*
   * in the case that JS_NFS_O_DIRECT has not been used, make compatible with Landmark's
   * SeisSpace environment variable SS_0_DIRECT
   */
   if (retval == NULL) {
   	 retval = getenv ("SS_O_DIRECT");
   }

  if (retval != NULL) {
    /*
     * JS_NFS_O_DIRECT environment variable is available
     */
  	for (k2 = 0; k2 < strlen(retval); k2++) {
  	  retval[k2] = toupper (retval[k2]);
  	}
  }
  return retval;
}

/* returns  1 if specified filename is a nfs file
 *          0 if specified filename is not a nfs file
 *         -1 if an error occurred
 *         -2 if filename is invalid or inaccessible
 */
int file_is_nfs (char *filename)
{
#if (LINUX || LINUXA || LINUXI || LINUXP)
/****************** Code using linux statfs ***************/

#ifndef NFS_SUPER_MAGIC
/* when including linux/nfs_fs.h was tried, it created compile bugs */
#define NFS_SUPER_MAGIC 0X6969
#endif

  struct statfs fs;

  if (statfs(filename,&fs) < 0) {
    printf ("file_is_nfs: error in statfs: %s\n", strerror(errno));
    if (errno == ENOENT) return (-2);
    return (-1);
  }
  if (fs.f_type == NFS_SUPER_MAGIC) {
  	if (DEBUG == 1) printf ("file_is_nfs: fs.f_type == %lx\n", (unsigned long) fs.f_type);
    return (1);
  }

#else
/*************** Code using solaris statvfs ******************/

  struct statvfs fs;

  if (statvfs(filename,&fs) < 0) {
    printf ("file_is_nfs: error in statvfs: %s\n", strerror(errno));
    if (errno == ENOENT) return (-2);
    return (-1);
  }
  if (strcmp(fs.f_basetype,"nfs") == 0) {
  	if (DEBUG == 1) printf ("file_is_nfs: fs.f_basetype == %s\n", fs.f_basetype); */
    return (1);
  }

#endif

  return (0);
}

/*
 * returns 0 if flags are able to be set (!=0) or cleared
 *   successfully
 */
int32_t set_fd_flags (int fd, int32_t flags, int set)
{
  int32_t err;
  /*
   * get the current flags
   */
  int32_t oldflags = (int32_t)fcntl (fd, F_GETFL);
  if (oldflags < 0) {
    printf ("set_fd_flags: F_GETFL error: %s\n", strerror(errno));
    return oldflags;
  }
  if (set != 0) {
    oldflags |= flags;
  }
  else {
    oldflags &= ~flags;
  }
  /*
   * set/unset the given given flags
   */
  err = (int32_t)fcntl (fd, F_SETFL, oldflags);
  return err;
}

/*
 * returns TRUE if the result of bitwise ANDing the given flags
 *   argument with the F_GETFL flags is > 0, otherwise FALSE
 */
int fd_flags_set (int fd, int32_t flags)
{
  int32_t retval = FALSE;
  /*
   * get the current flags
   */
  int32_t oldflags = (int32_t)fcntl (fd, F_GETFL);
  if (oldflags > -1) {
    if ((oldflags & flags) > 0) {
  	  retval = TRUE;
    }
    else {
  	  retval = FALSE;
    }
  }
  return retval;
}
#endif/*!_WIN32*/


/*////////////////////////////////////////////////////////////////////////////
// Method to get available disk space.
//
// The Java method File.getUsableSpace() does not give the correct result 
// for NFS-mounted filesystems on certain filers e.g. NetApps when running 
// RHEL 4 or 5 (WS 4.4 or 4.5).  The problem appears to be fixed 
// for RHEL 6.
////////////////////////////////////////////////////////////////////////////*/

/*
 * Return number of bytes in filesystem or -1 on error.
 */
long long jsutilfile_diskCapacityBytes(const char *file_sys)
{
#ifdef _WIN32
 BOOL status;
 ULARGE_INTEGER lpFreeBytesAvailable;
 ULARGE_INTEGER lpTotalNumberOfBytes;
 ULARGE_INTEGER lpTotalNumberOfFreeBytes;

 lpTotalNumberOfBytes.QuadPart  = 0L;
 status = GetDiskFreeSpaceEx(TEXT(file_sys),
		 &lpFreeBytesAvailable,
		 &lpTotalNumberOfBytes,
		 &lpTotalNumberOfFreeBytes);

 if (status == 0) {
	 lpTotalNumberOfBytes.QuadPart =
         ~lpTotalNumberOfBytes.QuadPart; /* fallback to say no limit */
  printf("GetDiskFreeSpaceEx(%s) failed code=%ld\n",file_sys, (long) GetLastError());
 }
 
  return lpTotalNumberOfBytes.QuadPart;

#else
#if defined(RS6000)
  struct statvfs vfs;
#else
#if (LINUX || LINUXA || LINUXI || LINUXP)
  struct statfs64 vfs;
#else
  struct statvfs64 vfs;
#endif
#endif
  int ret_stat;
  off64_t bcnt,bsize;
  off64_t nbytes = -1;

  errno = 0;

#if defined(RS6000)
  ret_stat = statvfs( (const char *)file_sys, &vfs );
#else
#if (LINUX || LINUXA || LINUXI || LINUXP)
  ret_stat = statfs64( (const char *)file_sys, &vfs );
#else
  ret_stat = statvfs64( (const char *)file_sys, &vfs );
#endif                                                          
#endif                                                          

  if (ret_stat != 0) {

    fprintf(stderr, "\n Failed to determine available disk space for: %s\n",
                    file_sys);

    perror(" jsutilfile_diskCapacityBytes:statvfs64 ");
    return nbytes;
  }

  bcnt  = (off64_t)vfs.f_bavail;

#if (LINUX || LINUXA || LINUXI || LINUXP)
  /* f_frsize gives a value that is too small on nfs-mounted filesystems

     bsize = (off64_t)vfs.f_frsize;
     if(bsize <= 0) bsize = (off64_t) vfs.f_bsize;
  */

  bsize = (off64_t) vfs.f_bsize;
  /* f_bsize may not always be filled in on earlier versions of linux -- try
   * f_frsize */
  if (bsize <= 0) bsize = (off64_t)vfs.f_frsize;

#else
  bsize = (off64_t)vfs.f_frsize;
  if (bsize <= 0) bsize = (off64_t)vfs.f_bsize;
#endif

  nbytes  = bcnt*bsize;


  return nbytes;
#endif
}

