/*
 * Decompiled with CFR 0.152.
 */
package org.javaseis.io;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.javaseis.io.IVirtualIO;
import org.javaseis.io.VirtualIO;
import org.javaseis.io.VirtualMappedIO;
import org.javaseis.util.SeisException;

public class TraceMap {
    private static final Logger LOG = Logger.getLogger(TraceMap.class.getName());
    private static final int CACHE_FLUSH_PERIOD = Integer.parseInt(System.getProperty("org.javaseis.io.tracemap.flush.seconds", "30"));
    private static final int BUFFER_SIZE = Integer.parseInt(System.getProperty("org.javaseis.io.tracemap.BufferSize", "200"));
    private final long[] _axisLengths;
    private final int[] _traceMapArray;
    private IVirtualIO _mapIO;
    private final ByteBuffer _mapBuffer;
    private final IntBuffer _mapBufferView;
    private final Map<Long, FoldItem> _cacheMap = new HashMap<Long, FoldItem>();
    private final ScheduledExecutorService ex = Executors.newSingleThreadScheduledExecutor();
    private final boolean _writeCaching;
    private SeisException _pendingException = null;
    private int _volumeIndex = Integer.MIN_VALUE;
    private long _readCacheHit;
    private long _readCounter;
    private long _putFoldCounter;
    private long _writeCounter;

    public TraceMap(long[] axisLengths, ByteOrder byteOrder, String path, String mode) throws SeisException {
        LOG.fine("TraceMap mode = " + mode);
        this._axisLengths = axisLengths;
        this._traceMapArray = new int[(int)this._axisLengths[2]];
        int bLength = 4 * (int)this._axisLengths[2];
        if (mode.compareToIgnoreCase("r") == 0) {
            this._mapIO = new VirtualMappedIO(path + File.separator + "TraceMap");
            this._writeCaching = false;
        } else if (mode.compareToIgnoreCase("wb") == 0) {
            this._writeCaching = true;
            LOG.fine("Buffered writing of TraceMap is enabled.");
            this._mapIO = new VirtualIO(path + File.separator + "TraceMap", mode);
            bLength = Math.max(bLength, 4 * BUFFER_SIZE);
        } else {
            LOG.fine("Buffered writing is not enabled");
            this._mapIO = new VirtualIO(path + File.separator + "TraceMap", mode);
            this._writeCaching = false;
        }
        byte[] b = new byte[bLength];
        this._mapBuffer = ByteBuffer.wrap(b);
        this._mapBuffer.order(byteOrder);
        this._mapBufferView = this._mapBuffer.asIntBuffer();
        this.initTraceMapArray();
        if (this._writeCaching) {
            int initialDelay = new Random().nextInt(CACHE_FLUSH_PERIOD);
            this.ex.scheduleAtFixedRate(new CacheWriter(this), initialDelay, CACHE_FLUSH_PERIOD, TimeUnit.SECONDS);
            LOG.config("Cache flush timer delay = " + initialDelay + " period = " + CACHE_FLUSH_PERIOD + " seconds");
        }
    }

    public synchronized int getFold(int[] position) throws SeisException {
        if (this._pendingException != null) {
            throw this._pendingException;
        }
        if (this._writeCaching) {
            long offset = this.getOffset(position);
            FoldItem item = this._cacheMap.get(offset);
            if (item != null) {
                ++this._readCacheHit;
                if (LOG.isLoggable(Level.FINER)) {
                    LOG.finer(this.hashCode() + " getFold = " + Arrays.toString(position) + " fold = " + this._traceMapArray[position[2]]);
                }
                return item.getFold(0);
            }
            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest(this.hashCode() + " Searching for " + Arrays.toString(position) + " but not found in cache");
            }
        }
        this.loadVolume(position);
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine(this.hashCode() + " getFold = " + Arrays.toString(position) + " fold = " + this._traceMapArray[position[2]]);
        }
        return this._traceMapArray[position[2]];
    }

    public void emptyCache() {
        this._volumeIndex = Integer.MIN_VALUE;
    }

    public synchronized void putFold(int[] position, int numTraces) throws SeisException {
        if (this._pendingException != null) {
            throw this._pendingException;
        }
        ++this._putFoldCounter;
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine(this.hashCode() + " putFold " + Arrays.toString(position) + " fold = " + numTraces);
        }
        if (!this._writeCaching) {
            this.doPutFold(position, numTraces);
            return;
        }
        long offset = this.getOffset(position);
        this._cacheMap.put(offset, new FoldItem(offset, position, numTraces));
        if (this._cacheMap.size() > BUFFER_SIZE) {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.fine("Cache has exceeded maximum size, flushing. Current cache depth = " + this.getCacheDepth());
            }
            this.flushCache();
        }
    }

    private synchronized void flushCache() throws SeisException {
        int origSize = this._cacheMap.size();
        if (origSize == 0) {
            return;
        }
        Set<Long> keys = this._cacheMap.keySet();
        ArrayList<Long> keyList = new ArrayList<Long>(keys);
        Collections.sort(keyList);
        FoldItem previous = this._cacheMap.get(keyList.get(0));
        for (int i = 1; i < origSize; ++i) {
            Long key = keyList.get(i);
            FoldItem item = this._cacheMap.get(key);
            if (previous.isContiguous(item)) {
                previous.addFold((Integer)item.fold.get(0));
                this._cacheMap.remove(key);
                continue;
            }
            previous = item;
        }
        int size = this._cacheMap.size();
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine(this.hashCode() + " Flushing TraceMap cache.  Original count = " + origSize + " count after consolidating contiquous frames = " + size);
        }
        for (FoldItem item : this._cacheMap.values()) {
            this.doPutFold(item.position, item.getFoldAll());
        }
        this._cacheMap.clear();
    }

    private void doPutFold(int[] position, int numTraces) throws SeisException {
        this.doPutFold(position, new int[]{numTraces});
    }

    private void doPutFold(int[] position, int[] fold) throws SeisException {
        long newMapFilePosition;
        ++this._writeCounter;
        int volumeIndex = this.getVolumeIndex(position);
        int frameIndex = position[2];
        long oldMapFilePosition = 4L * (this._axisLengths[2] * (long)volumeIndex + (long)frameIndex);
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest(this.hashCode() + " Writing to disk at offset " + oldMapFilePosition + " for volume index " + this._volumeIndex + " number of elements = " + fold.length);
        }
        if ((newMapFilePosition = this._mapIO.setPosition(oldMapFilePosition)) != oldMapFilePosition) {
            throw new SeisException("Unable to seek to file offset: " + oldMapFilePosition);
        }
        int start = position[2];
        int len = fold.length;
        if (start + len > this._traceMapArray.length - 1) {
            len = this._traceMapArray.length - start;
        }
        System.arraycopy(fold, 0, this._traceMapArray, start, len);
        this._mapBuffer.clear();
        this._mapBufferView.clear();
        this._mapBufferView.put(fold);
        this._mapBuffer.position(0);
        this._mapBuffer.limit(4 * fold.length);
        this._mapIO.write(this._mapBuffer);
    }

    private void initTraceMapArray() {
        Arrays.fill(this._traceMapArray, Integer.MIN_VALUE);
    }

    public void intializeTraceMapOnDisk() throws SeisException {
        int[] position = new int[this._axisLengths.length];
        for (int i = 1; i < position.length; ++i) {
            position[i] = (int)this._axisLengths[i] - 1;
        }
        this.putFold(position, 0);
        this.flushCache();
        this.initTraceMapArray();
        try {
            this._mapIO.flush();
        }
        catch (IOException e) {
            throw new SeisException("Error flushing map to disk: ", e);
        }
    }

    public void loadVolume(int[] position) throws SeisException {
        long newMapFilePosition;
        int volumeIndex = this.getVolumeIndex(position);
        if (this._volumeIndex == volumeIndex) {
            ++this._readCacheHit;
            return;
        }
        ++this._readCounter;
        this._mapBuffer.clear();
        this._mapBuffer.position(0);
        this._mapBufferView.position(0);
        this._mapBuffer.limit((int)(4L * this._axisLengths[2]));
        long oldMapFilePosition = 4L * this._axisLengths[2] * (long)volumeIndex;
        if (LOG.isLoggable(Level.FINER)) {
            LOG.finer(this.hashCode() + " Reading from disk at offset " + oldMapFilePosition + " for volume index " + volumeIndex);
        }
        if ((newMapFilePosition = this._mapIO.setPosition(oldMapFilePosition)) != oldMapFilePosition) {
            throw new SeisException("Unable to seek to file offset: " + oldMapFilePosition);
        }
        int numBytes = this._mapIO.read(this._mapBuffer);
        if ((long)numBytes != 4L * this._axisLengths[2]) {
            this.initTraceMapArray();
        } else {
            this._mapBufferView.get(this._traceMapArray);
        }
        this._volumeIndex = volumeIndex;
    }

    private int getVolumeIndex(int[] position) throws SeisException {
        int volumeIndex = -1;
        if (this._axisLengths.length == 3) {
            volumeIndex = 0;
        } else if (this._axisLengths.length == 4) {
            volumeIndex = position[3];
        } else if (this._axisLengths.length == 5) {
            long volumesPerHypberCube = this._axisLengths[3];
            long hcOffset = (long)position[4] * volumesPerHypberCube;
            volumeIndex = position[3];
            volumeIndex = (int)((long)volumeIndex + hcOffset);
        }
        if (volumeIndex < 0) {
            throw new SeisException("Invalid volume index: " + volumeIndex);
        }
        return volumeIndex;
    }

    private long getOffset(int[] position) throws SeisException {
        int volumeIndex = this.getVolumeIndex(position);
        int frameIndex = position[2];
        long pos = 4L * (this._axisLengths[2] * (long)volumeIndex + (long)frameIndex);
        return pos;
    }

    public int[] getTraceMapArray() {
        return this._traceMapArray;
    }

    protected void assertAllValuesInitialized() {
        for (int frameIndex = 0; frameIndex < this._traceMapArray.length; ++frameIndex) {
            if (this._traceMapArray[frameIndex] >= 1) continue;
            throw new RuntimeException("Invalid trace map value (" + this._traceMapArray[frameIndex] + ") found at frame index " + frameIndex);
        }
    }

    public IVirtualIO getMapIO() {
        return this._mapIO;
    }

    private int getCacheDepth() {
        return this._cacheMap.size();
    }

    private void putException(SeisException e) {
        this._pendingException = e;
    }

    public synchronized void close() throws SeisException {
        if (this._mapIO == null) {
            LOG.warning("Close has already been called on this object");
            return;
        }
        this.ex.shutdown();
        if (this._pendingException != null) {
            throw this._pendingException;
        }
        this.flushCache();
        LOG.fine(this.hashCode() + " TraceMap reads = " + this._readCounter + " read cached hits " + this._readCacheHit + " TraceMap putFold count = " + this._putFoldCounter + " TraceMap write count = " + this._writeCounter);
        this._mapIO.close();
        this._mapIO = null;
    }

    public void trackTime(boolean flag) {
        this._mapIO.trackTime(flag);
    }

    public long getIoBytes() {
        return this._mapIO.getIoBytes();
    }

    public float getIoTime() {
        return this._mapIO.getIoTime();
    }

    static {
        LOG.config("TraceMap cache buffer size  = " + BUFFER_SIZE);
        LOG.config("TraceMap cache will be flushed to disk every = " + CACHE_FLUSH_PERIOD + " seconds");
    }

    private static final class CacheWriter
    implements Runnable {
        private final TraceMap _parent;

        public CacheWriter(TraceMap parent) {
            this._parent = parent;
        }

        @Override
        public void run() {
            int depth = this._parent.getCacheDepth();
            if (depth == 0) {
                return;
            }
            LOG.fine("Flushing cache, current cache depth = " + depth);
            try {
                this._parent.flushCache();
            }
            catch (SeisException e) {
                this._parent.putException(e);
            }
        }
    }

    private static final class FoldItem {
        private final long offset;
        private final int[] position;
        private final ArrayList<Integer> fold = new ArrayList();

        public FoldItem(long offset, int[] position, int fold) {
            this.offset = offset;
            this.position = (int[])position.clone();
            this.fold.add(fold);
        }

        public void addFold(int fold) {
            this.fold.add(fold);
        }

        public int[] getFoldAll() {
            int[] a = new int[this.fold.size()];
            for (int i = 0; i < a.length; ++i) {
                a[i] = this.fold.get(i);
            }
            return a;
        }

        public int getFold(int index) {
            return this.fold.get(index);
        }

        public boolean isContiguous(FoldItem nextItem) {
            if (this.offset + 4L * (long)this.fold.size() == nextItem.offset) {
                return this.fold.size() != BUFFER_SIZE;
            }
            return false;
        }
    }
}

