/*
 * Decompiled with CFR 0.152.
 */
package org.javaseis.wavelet;

import java.util.Arrays;
import org.javaseis.fft.SeisFft;
import org.javaseis.parallel.DistributedArray;
import org.javaseis.parallel.DistributedArrayPositionIterator;
import org.javaseis.parallel.UniprocessorContext;
import org.javaseis.util.ArrayUtil;
import org.javaseis.wavelet.Window1D;

public class WindowedFourierTransform {
    private int nwin;
    private int windowLength;
    private int nf;
    private int if1;
    private int ifn;
    private int nt;
    private int ntw;
    private int nfw;
    private float dt;
    private float df;
    private SeisFft tfft;
    private int nfft;
    private int nyq;
    private Window1D win;
    private int maxlen;

    public WindowedFourierTransform(int numSamples, float sampleRate, float windowFftLength, float windowOverlap, float fmin, float fmax) {
        Window1D window = new Window1D(numSamples, sampleRate, windowFftLength - 1.0f, windowOverlap);
        this.initializeWindowedFourierTransform(window, numSamples, sampleRate, fmin, fmax);
    }

    public WindowedFourierTransform(Window1D window, int numSamples, float sampleRate, float fmin, float fmax) {
        this.initializeWindowedFourierTransform(window, numSamples, sampleRate, fmin, fmax);
    }

    public void initializeWindowedFourierTransform(Window1D window, int numSamples, float sampleRate, float fmin, float fmax) {
        this.nt = numSamples;
        this.win = window;
        this.nwin = this.win.getWindowCount();
        this.windowLength = this.win.getWindowLength();
        this.tfft = new SeisFft(this.windowLength, 0.0f);
        this.nfft = this.tfft.getLength();
        if (this.nfft != this.windowLength) {
            throw new IllegalArgumentException("Window length " + this.windowLength + " does not match FFT length " + this.nfft);
        }
        this.nyq = 1 + this.nfft / 2;
        this.dt = sampleRate / 1000.0f;
        this.df = 1.0f / (this.dt * (float)this.tfft.getLength());
        this.if1 = (int)Math.rint(fmin / this.df);
        if (this.if1 < 0) {
            this.if1 = 0;
        }
        this.ifn = (int)Math.rint(fmax / this.df);
        if (this.ifn <= this.if1 || this.ifn >= this.nyq) {
            this.ifn = this.nyq - 1;
        }
        this.nf = this.ifn - this.if1 + 1;
        this.nfw = this.nwin * this.nf;
        this.ntw = this.nwin * this.nt;
        this.maxlen = Math.max(2 * this.nfw, this.ntw);
    }

    public int getTimeSamples() {
        return this.nt;
    }

    public int getTransformLength() {
        return this.tfft.getLength();
    }

    public float getSampleFrequency() {
        return this.df;
    }

    public int getFrequencyCount() {
        return this.nf;
    }

    public int getStartFrequencyIndex() {
        return this.if1;
    }

    public int getWindowedTransformLength() {
        return 2 * this.nfw;
    }

    public float[][] forwardTransform(float[] trc) {
        float[][] wft = new float[this.nwin][2 * this.nf];
        this.forwardTransform(trc, wft);
        return wft;
    }

    public void forwardTransform(float[] trc, float[][] wft) {
        assert (trc.length >= this.nt) : "Input array length is too short";
        assert (wft.length >= this.nwin && wft[0].length >= 2 * this.nf) : "Output array size does not match windowed FFT size";
        float[] c = new float[this.nfft + 2];
        for (int j = 0; j < this.nwin; ++j) {
            this.win.getWindowSamples(trc, j, c);
            Arrays.fill(c, this.windowLength, this.nfft + 2, 0.0f);
            this.tfft.realToComplex(c);
            ArrayUtil.arraycopy(c, 2 * this.if1, wft[j], 0, 2 * this.nf);
        }
    }

    public void forwardTransform(float[] trc, float[] wft) {
        assert (trc.length >= this.nt) : "Input array length is too short";
        assert (wft.length >= this.maxlen) : "Output array size does not match windowed FFT size";
        int k = 0;
        float[] c = new float[this.nfft + 2];
        for (int j = 0; j < this.nwin; ++j) {
            this.win.getWindowSamples(trc, j, c);
            Arrays.fill(c, this.nt, 2 * this.nfft, 0.0f);
            this.tfft.realToComplex(c);
            ArrayUtil.arraycopy(c, 2 * this.if1, wft, k, 2 * this.nf);
            k += 2 * this.nf;
        }
    }

    public void forwardTransform(DistributedArray da) {
        float[] trc = new float[this.nt];
        float[] ctrc = new float[2 * this.nfw];
        int nx = da.getLength(1);
        int ny = da.getLength(2);
        DistributedArray r = da.distributedView();
        da.setElementCount(2);
        da.setShape(new int[]{this.nf * this.nwin, nx, ny});
        int idir = -1;
        int[] position = new int[]{0, 0, 0};
        DistributedArrayPositionIterator dapi = new DistributedArrayPositionIterator(r, position, idir);
        while (dapi.hasNext()) {
            dapi.next();
            r.getTrace(trc, position);
            this.forwardTransform(trc, ctrc);
            da.putTrace(ctrc, position);
        }
    }

    public void inverseTransform(float[][] wft, float[] trc) {
        assert (wft.length >= this.nwin && wft[0].length >= 2 * this.nf) : "Input array size does not match windowed FFT size";
        assert (trc.length >= this.nt) : "Output array length is too short";
        float[] c = new float[this.nfft + 2];
        for (int j = 0; j < this.nwin; ++j) {
            Arrays.fill(c, 0, 2 * this.if1, 0.0f);
            ArrayUtil.arraycopy(wft[j], 0, c, 2 * this.if1, 2 * this.nf);
            Arrays.fill(c, 2 * (this.ifn + 1), this.nfft + 2, 0.0f);
            this.tfft.complexToReal(c);
            this.win.putWindowSamples(c, j, trc);
        }
    }

    public void inverseTransform(float[] ctrc, float[] trc) {
        assert (ctrc.length >= this.maxlen) : "Input array size does not match windowed FFT size";
        assert (trc.length >= this.nt) : "Output array length is too short";
        int k = 0;
        float[] c = new float[this.nfft + 2];
        for (int j = 0; j < this.nwin; ++j) {
            Arrays.fill(c, 0, 2 * this.if1, 0.0f);
            ArrayUtil.arraycopy(ctrc, k, c, 2 * this.if1, 2 * this.nf);
            Arrays.fill(c, 2 * this.ifn, this.nfft + 2, 0.0f);
            this.tfft.complexToReal(c);
            this.win.putWindowSamples(c, j, trc);
            k += this.windowLength;
        }
    }

    public void inverseTransform(DistributedArray da) {
        float[] trc = new float[this.ntw];
        float[] ctrc = new float[2 * this.nfw];
        float[] cw = new float[2 * this.nyq];
        int nx = da.getLength(1);
        int ny = da.getLength(2);
        DistributedArray c = da.distributedView();
        c.setShape(new int[]{this.nf * this.nwin, nx, ny});
        da.setElementCount(1);
        da.setShape(new int[]{this.nt, nx, ny});
        int idir = 1;
        int[] position = new int[]{0, 0, 0};
        DistributedArrayPositionIterator dapi = new DistributedArrayPositionIterator(c, position, idir);
        while (dapi.hasNext()) {
            dapi.next();
            c.getTrace(ctrc, position);
            int k = 0;
            for (int j = 0; j < this.nwin; ++j) {
                Arrays.fill(cw, 0, 2 * this.if1, 0.0f);
                ArrayUtil.arraycopy(ctrc, k, cw, 2 * this.if1, 2 * this.nf);
                Arrays.fill(cw, 2 * this.ifn, 2 * this.nyq, 0.0f);
                this.tfft.complexToReal(cw);
                this.win.putWindowSamples(cw, j, trc);
                k += this.windowLength;
            }
            da.putTrace(trc, position);
        }
    }

    public static void main(String[] args) {
        int i;
        UniprocessorContext pc = new UniprocessorContext();
        int nx = 20;
        int ny = 15;
        int nt = 16;
        float sr = 1.0f;
        float padt = 0.0f;
        float fmin = 0.0f;
        float fmax = 0.0f;
        int windowLen = 8;
        int windowOvl = 4;
        float[] x = new float[nt];
        float[] xo = (float[])x.clone();
        Arrays.fill(x, 1.0f);
        System.out.println("WindowedFourierTransform Tests");
        System.out.println("\nSimple test with all ones:");
        Window1D w1d = new Window1D(nt, windowLen, windowOvl, 0, 0);
        System.out.println("WindowLength " + w1d.getWindowLength());
        System.out.println("WindowCount " + w1d.getWindowCount());
        System.out.println("WindowOverlap " + w1d.getWindowOverlap());
        WindowedFourierTransform wft = new WindowedFourierTransform(w1d, nt, sr, fmin, fmax);
        System.out.println("Before forwardTransform x = \n" + Arrays.toString(x));
        float[][] cwx = wft.forwardTransform(x);
        System.out.println(" After forwardTransform cwx = ");
        for (i = 0; i < w1d.getWindowCount(); ++i) {
            System.out.println(Arrays.toString(cwx[i]));
        }
        wft.inverseTransform(cwx, xo);
        System.out.println("After inverseTransform xo = \n" + Arrays.toString(xo));
        for (i = 0; i < nt; ++i) {
            assert ((double)Math.abs(xo[i] - 1.0f) < 1.0E-6) : "Reconstructed value is out of range at index " + i;
        }
        System.out.println("\nRandomized Tests");
        for (i = 0; i < nt; ++i) {
            x[i] = (float)Math.random();
            xo[i] = 0.0f;
        }
        System.out.println("Before forwardTransform x = \n" + Arrays.toString(x));
        cwx = wft.forwardTransform(x);
        System.out.println(" After forwardTransform cwx = ");
        for (i = 0; i < w1d.getWindowCount(); ++i) {
            System.out.println(Arrays.toString(cwx[i]));
        }
        wft.inverseTransform(cwx, xo);
        System.out.println("After inverseTransform xo = \n" + Arrays.toString(xo));
        for (i = 0; i < nt; ++i) {
            assert ((double)Math.abs(xo[i] - x[i]) < 1.0E-6) : "Reconstructed value is out of range at index " + i;
        }
        System.out.println("*** Window1D Success ***");
    }
}

