/*
 * Decompiled with CFR 0.152.
 */
package org.javaseis.seiszip;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.util.logging.Logger;
import org.javaseis.seiszip.HdrCompressor;
import org.javaseis.seiszip.SeisPEG;

public class SegyZip {
    private static final Logger LOG = Logger.getLogger("org.javaseis.seiszip");
    static final int LEN_REEL_HDR = 3200;
    static final int LEN_BINARY_HDR = 400;
    static final int NBYTES_PER_HDR = 240;
    static final int NINTS_PER_HDR = 60;
    static final int COOKIE_V1 = 35478349;
    private FileInputStream _inputStream;
    private FileOutputStream _outputStream;
    private FileChannel _inputChannel;
    private FileChannel _outputChannel;
    private long _nbytesRead = 0L;
    private long _nbytesWritten = 0L;
    private SeisPEG _seisPEG;
    private int _nbytesPerTrace;
    private int _frameKeyOffset;
    private int _frameKeyLength;
    private float[][] _traces;
    private int[][] _hdrs;
    private byte[] _compressedHdrBytes;
    private byte[] _compressedTrcBytes;
    private ByteBuffer _compressedHdrByteBuffer;
    private ByteBuffer _compressedTrcByteBuffer;
    private long _tracesToDump;
    private ByteBuffer _workBuffer;
    private long _frameWrittenCount = 0L;

    public SegyZip(String inFile, int frameKeyOffset, int frameKeyLength, long tracesToDump) throws IOException {
        float distortion = 0.1f;
        float ftGainExponent = 0.0f;
        int verticalBlockSize = 64;
        int horizontalBlockSize = 64;
        int verticalTransLength = 16;
        int horizontalTransLength = 16;
        this.init(inFile, null, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, frameKeyOffset, frameKeyLength, tracesToDump);
    }

    public SegyZip(String inFile, String outFile, float distortion, float ftGainExponent, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength, int frameKeyOffset, int frameKeyLength) throws IOException {
        long tracesToDump = -1L;
        this.init(inFile, outFile, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, frameKeyOffset, frameKeyLength, tracesToDump);
    }

    private void init(String inFile, String outFile, float distortion, float ftGainExponent, int verticalBlockSize, int horizontalBlockSize, int verticalTransLength, int horizontalTransLength, int frameKeyOffset, int frameKeyLength, long tracesToDump) throws IOException {
        this._frameKeyOffset = frameKeyOffset;
        this._frameKeyLength = frameKeyLength;
        this._tracesToDump = tracesToDump;
        this._inputStream = new FileInputStream(inFile);
        this._inputChannel = this._inputStream.getChannel();
        if (outFile != null) {
            this._outputStream = new FileOutputStream(outFile);
            this._outputChannel = this._outputStream.getChannel();
        }
        this._workBuffer = ByteBuffer.allocate(8);
        this._workBuffer.order(ByteOrder.BIG_ENDIAN);
        if (this._outputChannel != null) {
            this._workBuffer.putInt(0, 35478349);
            this._workBuffer.putInt(4, 0);
            this._workBuffer.position(0);
            int nWritten = this._outputChannel.write(this._workBuffer);
            this._nbytesWritten += (long)nWritten;
            if (nWritten != 8) {
                throw new IOException("Error writing file header: " + nWritten + "!=8");
            }
        }
        ByteBuffer reelHdrBuffer = ByteBuffer.allocate(3200);
        int nRead = this._inputChannel.read(reelHdrBuffer);
        this._nbytesRead += (long)nRead;
        if (nRead != 3200) {
            throw new IOException("Error reading SEG-Y reel header: " + nRead + "!=" + 3200);
        }
        if (outFile != null) {
            reelHdrBuffer.position(0);
            int nWritten = this._outputChannel.write(reelHdrBuffer);
            this._nbytesWritten += (long)nWritten;
            if (nWritten != 3200) {
                throw new IOException("Error writing SEG-Y reel header: " + nWritten + "!=" + 3200);
            }
        }
        ByteBuffer binaryHdrBuffer = ByteBuffer.allocate(400);
        nRead = this._inputChannel.read(binaryHdrBuffer);
        this._nbytesRead += (long)nRead;
        if (nRead != 400) {
            throw new IOException("Error reading SEG-Y binary header: " + nRead + "!=" + 400);
        }
        if (outFile != null) {
            binaryHdrBuffer.position(0);
            int nWritten = this._outputChannel.write(binaryHdrBuffer);
            this._nbytesWritten += (long)nWritten;
            if (nWritten != 400) {
                throw new IOException("Error writing SEG-Y binary header: " + nWritten + "!=" + 400);
            }
        }
        binaryHdrBuffer.order(ByteOrder.BIG_ENDIAN);
        short tracesPerFrame = binaryHdrBuffer.getShort(12);
        System.out.println("tracesPerFrame= " + tracesPerFrame);
        float sampleInterval = (float)binaryHdrBuffer.getShort(16) / 1000.0f;
        System.out.println("sampleInterval= " + sampleInterval);
        short samplesPerTrace = binaryHdrBuffer.getShort(20);
        System.out.println("samplesPerTrace= " + samplesPerTrace);
        this._nbytesPerTrace = samplesPerTrace * 4;
        this._seisPEG = new SeisPEG(samplesPerTrace, tracesPerFrame, distortion, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength);
        if ((double)ftGainExponent != 0.0) {
            this._seisPEG.setGainExponent(ftGainExponent);
        }
        this._traces = new float[tracesPerFrame][samplesPerTrace];
        this._hdrs = new int[tracesPerFrame][60];
        this._compressedHdrBytes = new byte[HdrCompressor.getOutputBufferSize(60, tracesPerFrame)];
        this._compressedHdrByteBuffer = ByteBuffer.wrap(this._compressedHdrBytes);
        this._compressedTrcBytes = new byte[samplesPerTrace * tracesPerFrame * 4];
        this._compressedTrcByteBuffer = ByteBuffer.wrap(this._compressedTrcBytes);
    }

    public void dump() throws IOException {
        this.zip();
    }

    public void zip() throws IOException {
        ByteBuffer hdrByteBuffer = ByteBuffer.allocateDirect(240);
        hdrByteBuffer.order(ByteOrder.BIG_ENDIAN);
        ByteBuffer trcByteBuffer = ByteBuffer.allocateDirect(this._nbytesPerTrace);
        trcByteBuffer.order(ByteOrder.BIG_ENDIAN);
        IntBuffer hdrBuffer = hdrByteBuffer.asIntBuffer();
        FloatBuffer trcBuffer = trcByteBuffer.asFloatBuffer();
        boolean finished = false;
        long traceInFileCount = 1L;
        int traceInFrameCount = 0;
        int frameKeyLast = Integer.MIN_VALUE;
        while (!finished) {
            hdrByteBuffer.position(0);
            int nRead = this._inputChannel.read(hdrByteBuffer);
            this._nbytesRead += (long)nRead;
            if (nRead == -1) {
                if (traceInFrameCount != 0) {
                    this.writeFrame(traceInFrameCount, traceInFileCount);
                }
                finished = true;
            } else {
                int frameKey;
                if (nRead != 240) {
                    throw new IOException("Error reading SEG-Y trace header " + traceInFileCount + ": " + nRead + "!=" + 240);
                }
                trcByteBuffer.position(0);
                nRead = this._inputChannel.read(trcByteBuffer);
                this._nbytesRead += (long)nRead;
                if (nRead != this._nbytesPerTrace) {
                    throw new IOException("Error reading SEG-Y trace " + traceInFileCount + ": " + nRead + "!=" + this._nbytesPerTrace);
                }
                if (this._frameKeyLength == 2) {
                    frameKey = hdrByteBuffer.getShort(this._frameKeyOffset);
                } else if (this._frameKeyLength == 4) {
                    frameKey = hdrByteBuffer.getInt(this._frameKeyOffset);
                } else {
                    throw new RuntimeException("Frame key length " + this._frameKeyLength + " is not recognized");
                }
                if (this._tracesToDump >= 0L) {
                    System.out.println("frameKey= " + frameKey + " at trace " + traceInFileCount);
                }
                if (frameKey != frameKeyLast) {
                    if (traceInFrameCount != 0) {
                        this.writeFrame(traceInFrameCount, traceInFileCount);
                    }
                    traceInFrameCount = 0;
                }
                frameKeyLast = frameKey;
                if (traceInFrameCount >= this._hdrs.length) {
                    throw new IOException("Observed traces per frame exceeds " + this._hdrs.length + " - the header at byte offset " + this._frameKeyOffset + " may be wrong");
                }
                hdrBuffer.position(0);
                hdrBuffer.get(this._hdrs[traceInFrameCount]);
                trcBuffer.position(0);
                trcBuffer.get(this._traces[traceInFrameCount]);
                ++traceInFrameCount;
            }
            if (this._tracesToDump < 0L || ++traceInFileCount < this._tracesToDump) continue;
            finished = true;
        }
        this._inputStream.close();
        if (this._outputStream != null) {
            this._outputStream.close();
        }
    }

    private void writeFrame(int traceInFrameCount, long traceInFileCount) throws IOException {
        if (traceInFrameCount == 0) {
            return;
        }
        if (this._tracesToDump >= 0L) {
            return;
        }
        if (traceInFrameCount == 1) {
            LOG.warning("Only one trace per frame found near trace " + traceInFileCount);
        }
        int nbytesHdrs = this._seisPEG.compressHdrs(this._hdrs, 60, traceInFrameCount, this._compressedHdrBytes);
        int nbytesTraces = this._seisPEG.compress(this._traces, traceInFrameCount, this._compressedTrcBytes);
        this._workBuffer.putInt(0, nbytesHdrs);
        this._workBuffer.putInt(4, nbytesTraces);
        this._workBuffer.position(0);
        int nWritten = this._outputChannel.write(this._workBuffer);
        this._nbytesWritten += (long)nWritten;
        if (nWritten != 8) {
            throw new IOException("Error writing mini header near trace " + traceInFileCount + ": " + nWritten + "!=8");
        }
        this._compressedHdrByteBuffer.position(0);
        this._compressedHdrByteBuffer.limit(nbytesHdrs);
        nWritten = this._outputChannel.write(this._compressedHdrByteBuffer);
        this._nbytesWritten += (long)nWritten;
        if (nWritten != nbytesHdrs) {
            throw new IOException("Error writing compressed headers near trace " + traceInFileCount + ": " + nWritten + "!=" + nbytesHdrs);
        }
        this._compressedTrcByteBuffer.position(0);
        this._compressedTrcByteBuffer.limit(nbytesTraces);
        nWritten = this._outputChannel.write(this._compressedTrcByteBuffer);
        this._nbytesWritten += (long)nWritten;
        if (nWritten != nbytesTraces) {
            throw new IOException("Error writing compressed data near trace " + traceInFileCount + ": " + nWritten + "!=" + nbytesTraces);
        }
        ++this._frameWrittenCount;
        if (this._frameWrittenCount % 1000L == 0L) {
            System.out.println("Finished compressing and writing frame " + this._frameWrittenCount + " ...");
        }
    }

    public double getCompressionRatio() {
        return (double)this._nbytesRead / (double)this._nbytesWritten;
    }

    public static void main(String[] args) {
        if (args.length == 4) {
            try {
                String inFile = args[0];
                int frameKeyOffset = Integer.parseInt(args[1]);
                int frameKeyLength = Integer.parseInt(args[2]);
                int tracesToDump = Integer.parseInt(args[3]);
                SegyZip segyZip = new SegyZip(inFile, frameKeyOffset, frameKeyLength, tracesToDump);
                segyZip.dump();
                System.exit(0);
            }
            catch (Exception e) {
                e.printStackTrace();
                System.exit(-1);
            }
        } else if (args.length == 8) {
            try {
                String inFile = args[0];
                String outFile = args[1];
                float distortion = Float.parseFloat(args[2]);
                float ftGainExponent = Float.parseFloat(args[3]);
                int verticalBlockSize = Integer.parseInt(args[4]);
                int horizontalBlockSize = Integer.parseInt(args[5]);
                int frameKeyOffset = Integer.parseInt(args[6]);
                int frameKeyLength = Integer.parseInt(args[7]);
                int verticalTransLength = 16;
                int horizontalTransLength = 16;
                SegyZip segyZip = new SegyZip(inFile, outFile, distortion, ftGainExponent, verticalBlockSize, horizontalBlockSize, verticalTransLength, horizontalTransLength, frameKeyOffset, frameKeyLength);
                segyZip.zip();
                System.out.println("Data compression ratio= " + segyZip.getCompressionRatio());
                System.exit(0);
            }
            catch (Exception e) {
                e.printStackTrace();
                System.exit(-1);
            }
        } else {
            throw new IllegalArgumentException("Usage: java SegyZip segyInputFile zippedOutputFile distortion ftGainExponent verticalBlockSize horizontalBlockSize frameKeyOffset frameKeyLength\nor\njava SegyZip segyInputFile frameKeyOffset frameKeyLength tracesToDump");
        }
    }
}

